import os
import jinja2
import logging
from contextlib import contextmanager
from sandbox import sdk2
from sandbox.projects.paysys.tasks.monitorings.PaysysMonitoringsApplier import helpers

# WARNING: This projects must be in sync with the projects in monitorings
# configs dir by hand, because there is no way to import configs to the
# sandbox task unfortunately.
PROJECTS = [
    "oplata",
    "oebs",
    "oebsapi",
    "paysys",
    "dashboards",
    "hyperion",
    "bi",
    "crm",
    "cloning",
    "configshop",
    "backup",
    "dist",
    "vertica",
    "oebsfunction",
    "balance",
    "balance_app",
    "market",
    "meta",
    "reports",
    "uzedo",
    "billing30",
    "trust",
    "dcsaap",
    "refs",
    "mdh",
    "yabank",
    "ift",
    "bcl",
    "dwh",
    "apikeys",
    "yb_log_tariff",
    "octopus",
    "balance_fop",
    "whitespirit",
    "diehard",
    "agency_rewards",
]


class PaysysMonitoringsBinary(sdk2.resource.AbstractResource):
    releasable = True
    releasers = [
        "PAYSYSADMIN",
    ]
    release_subscribers = ["PAYSYSADMIN"]


class PaysysMonitoringsApplicationResultStdout(sdk2.resource.AbstractResource):
    releasable = True
    releasers = [
        "PAYSYSADMIN",
    ]
    release_subscribers = ["PAYSYSADMIN"]


class PaysysMonitoringsApplicationResultStderr(sdk2.resource.AbstractResource):
    releasable = True
    releasers = [
        "PAYSYSADMIN",
    ]
    release_subscribers = ["PAYSYSADMIN"]


class PaysysMonitoringsApplier(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 512
        cores = 1

    class Parameters(sdk2.Parameters):
        description = "Apply juggler monitorings for selected projects"

        with sdk2.parameters.CheckGroup("Projects") as projects:
            for option in PROJECTS:
                projects.values[option] = projects.Value(option, checked=False)

        dry_run = sdk2.parameters.Bool("Dry run", default=True)

        binary = sdk2.parameters.Resource(
            "Monitorings binary",
            resource_type=PaysysMonitoringsBinary,
            required=True,
        )

        conductor_token = sdk2.parameters.String(
            "Conductor token vault item",
            default="robot-ps-prod-ci-conductor-oauth",
            required=True,
        )

        juggler_token = sdk2.parameters.String(
            "Juggler token vault item",
            default="robot-ps-prod-ci-juggler-oauth",
            required=True,
        )

        solomon_token = sdk2.parameters.String(
            "Solomon token vault item",
            default="robot-ps-prod-ci-solomon-oauth",
            required=True,
        )

    @contextmanager
    def _set_secrets(self):
        conductor_token_value = sdk2.task.VaultItem(
            self.owner,
            self.Parameters.conductor_token,
        )
        conductor_token = conductor_token_value.data()

        juggler_token_value = sdk2.task.VaultItem(
            self.owner,
            self.Parameters.juggler_token,
        )
        juggler_token = juggler_token_value.data()

        solomon_token_value = sdk2.task.VaultItem(
            self.owner,
            self.Parameters.solomon_token,
        )
        solomon_token = solomon_token_value.data()

        os.environ["CONDUCTOR_TOKEN"] = conductor_token
        os.environ["JUGGLER_TOKEN"] = juggler_token
        os.environ["SOLOMON_TOKEN"] = solomon_token

        yield

        del os.environ["CONDUCTOR_TOKEN"]
        del os.environ["JUGGLER_TOKEN"]
        del os.environ["SOLOMON_TOKEN"]

    def _run_command(self, command, project):
        logging.info("Trying to run command " + command)

        with sdk2.helpers.ProcessLog(self, logger='monitorings_apply_' + project) as log:
            sdk2.helpers.subprocess.check_call(
                command.split(),
                stdout=log.stdout,
                stderr=log.stderr,
            )

    def _create_resource_from_result(self, project):
        logs_resource = sdk2.service_resources.TaskLogs.find(task=self).first()

        log_filename_stderr = "monitorings_apply_{}.err.log".format(project)

        resource_stderr = sdk2.ResourceData(
            PaysysMonitoringsApplicationResultStderr(
                self,
                "monitorings_application_result_stderr_" + project,
                log_filename_stderr + ".html",
            )
        )

        with open(str(logs_resource.path) + "/" + log_filename_stderr) as logs_resource_file:
            result_html = helpers.log_to_colored_html(
                logs_resource_file.readlines()
            )

            resource_stderr.path.write_bytes(result_html)

        resource_stderr.ready()

        log_filename_stdout = "monitorings_apply_{}.out.log".format(project)

        resource_stdout = sdk2.ResourceData(
            PaysysMonitoringsApplicationResultStdout(
                self,
                "monitorings_application_result_stdout_" + project,
                log_filename_stdout + ".html",
            )
        )

        with open(str(logs_resource.path) + "/" + log_filename_stdout) as logs_resource_file:
            result_html = helpers.log_to_colored_html(
                logs_resource_file.readlines()
            )

            resource_stdout.path.write_bytes(result_html)

        resource_stdout.ready()

        return resource_stderr, resource_stdout

    def _gen_result(self, result_resource_stderr, result_resource_stdout, project):
        template = {
            "result": """<b>Note: For color html output check the attached resources.</b>

<b>Application result for project {{ project }}:</b>

STDERR:
{{ result_stderr }}

STDOUT:
{{ result_stdout }}
"""
        }
        env = jinja2.Environment(loader=jinja2.DictLoader(template))
        with open(str(result_resource_stderr.path)) as result_stderr_file:
            with open(str(result_resource_stdout.path)) as result_stdout_file:
                return env.get_template("result").render(
                    result_stderr=result_stderr_file.read(),
                    result_stdout=result_stdout_file.read(),
                    project=project,
                )

    def on_execute(self):
        monitorings_bin_res = self.Parameters.binary

        with self._set_secrets():
            for project in self.Parameters.projects:
                command = "{} --project {} {}".format(
                    sdk2.ResourceData(monitorings_bin_res).path,
                    project,
                    "--dry" if self.Parameters.dry_run else "",
                )
                self._run_command(command, project)
                result_resource_stderr, result_resource_stdout = self._create_resource_from_result(project)
                self.set_info(
                    self._gen_result(result_resource_stderr, result_resource_stdout, project),
                    do_escape=False,
                )
