import logging
import os
import subprocess

from sandbox import sdk2
from sandbox.projects.yabs.qa.module_base import ModuleBase
from sandbox.projects.pcode.qa.utils import wait_ping

from sandbox.projects.common.vcs import arc


class ApphostSUT(ModuleBase):
    def __init__(self, adapter):
        ModuleBase.__init__(self, adapter)
        self._active = False

        self._ports = dict()
        self._ports['apphost_admin_port'] = self.adapter.get_apphost_port()
        self._ports['apphost_port'] = self._ports['apphost_admin_port'] + 4

        self._tokens = self.adapter.get_tokens()
        os.environ.update(self._tokens)

        self._mount_arc()
        self._build_apphost()
        self._change_binaries()

    def __del__(self):
        if hasattr(self, 'local_arcadia'):
            self.local_arcadia.__del__()

    def _mount_arc(self):
        logging.info("Getting local arcadia")
        arc_client = arc.Arc()

        try:
            if self.adapter.get_apphost_mode() == "arcadia" and self.adapter.get_arcadia_revision() != "trunk":
                self.local_arcadia = arc_client.mount_path(
                    path='',
                    changeset=self.adapter.get_arcadia_revision(),
                    fetch_all=False
                )
            else:
                # trunk
                logging.info("Trunk")
                self.local_arcadia = arc_client.mount_path(None, None, fetch_all=False)

            logging.info("Enter local arcadia path")
            self.local_arcadia_path = self.local_arcadia.__enter__()
        except Exception as e:
            logging.error(str(e))
            self.local_arcadia_path = os.path.abspath(os.path.join(os.getcwd(), '../../../..'))

        logging.info("Getting 'ya' path")
        self.ya_path = os.path.join(self.local_arcadia_path, 'ya')
        if not os.path.exists(self.ya_path):
            self.ya_path = os.path.join(self.local_arcadia_path, 'devtools', 'ya', 'ya')

    def _build_apphost(self):
        # build with -D param (download latest released binaries)
        cmd = [
            self.ya_path,
            'tool',
            'apphost',
            'setup',
            '-y',
            '-D',
            '--local-arcadia-path',
            self.local_arcadia_path,
            self.adapter.get_apphost_mode(),
            '--vertical',
            self.adapter.get_apphost_vertical(),
        ]
        logging.info('Build dir and download binaries from Sandbox: {}'.format(cmd))

        self._apphost_build_log_context = sdk2.helpers.ProcessLog(self.adapter.task_instance, 'apphost.build')
        self._apphost_build_log_context.__enter__()
        subprocess.Popen(
            cmd, stdout=self._apphost_build_log_context.stdout, stderr=self._apphost_build_log_context.stderr
        )

    def _change_binaries(self):
        pass

    def _run_apphost(self):
        cmd = [
            self.ya_path,
            'tool',
            'apphost',
            'setup',
            '-y',
            '-S',
            '-D',
            '--local-arcadia-path',
            self.local_arcadia_path,
            '-P',
            str(self._ports["apphost_admin_port"]),
            self.adapter.get_apphost_mode(),
            '--vertical',
            self.adapter.get_apphost_vertical(),
        ]
        logging.info('Running apphost with command: {}'.format(cmd))

        self._apphost_process_log_context = sdk2.helpers.ProcessLog(self.adapter.task_instance, 'apphost.run')
        self._apphost_process_log_context.__enter__()
        self._apphost_process = subprocess.Popen(
            cmd, stdout=self._apphost_process_log_context.stdout, stderr=self._apphost_process_log_context.stderr
        )

    def _stop_apphost(self, *args):
        self._apphost_process.kill()
        self._apphost_process.communicate()
        self._apphost_process_log_context.__exit__(*args)
        self._apphost_process_log_context = None

    def __enter__(self):
        self._run_apphost()

        self._active = True
        wait_ping(
            "http://localhost:{}/admin?action=ping".format(self._ports['apphost_admin_port']),
            timeout=20*60,
            max_pause=5*60
        )

        return self

    def __exit__(self, *args):
        logging.info('Exiting apphost')
        self._stop_apphost(*args)
        self._active = False

    def get_port(self):
        if self._active:
            return self._ports['apphost_port']
        else:
            return None
