import logging
import os
import subprocess

import patch_renderer

from sandbox import sdk2
from sandbox.common.fs import get_unique_file_name
from sandbox.projects.common.yabs.cachedaemon import CacheDaemonStubSandboxNonIntegrated
from sandbox.projects.yabs.qa.module_base import ModuleBase

from sandbox.projects.pcode.qa.utils import wait_ping


class SsrSUT(ModuleBase):

    def __init__(self, adapter):
        ModuleBase.__init__(self, adapter)
        self._active = False

        self._ports = self.adapter.get_ports_map()

        self._cachedaemon = self._cachedaemon = CacheDaemonStubSandboxNonIntegrated(
            cache_daemon_executable_path=self.adapter.get_cache_daemon_executable_path(),
            dump_path=self.adapter.get_cache_daemon_stub_path(),
            data_dir=get_unique_file_name(self.adapter.get_work_dir(), 'cache_daemon_ssr_data'),
            log_subdir=get_unique_file_name(self.adapter.get_logs_dir(), 'cache_daemon_ssr_logs'),
            start_on_creation=False,
            key_header=self.adapter.get_cache_daemon_key_headers(),
            services=None,
            services_ports=self._ports.get('cache_daemon_ports'),
        )

        self._unpack_renderer()
        self._patch_archon_renderer()  # TODO Delete after https://st.yandex-team.ru/PCODE-20328

    def _unpack_renderer(self):
        self._pcode_unpacked_path = os.path.join(self.adapter.get_work_dir(), 'renderer')

        logging.info('Unpacking pcode resource to {}'.format(self._pcode_unpacked_path))
        with sdk2.helpers.ProcessLog(self.adapter.task_instance, 'renderer.unpack') as process_log:
            subprocess.Popen(
                ['tar', '-xvf', self.adapter.get_pcode_resource_path(), '-C', self.adapter.get_work_dir()],
                stdout=process_log.stdout,
                stderr=process_log.stderr,
            ).wait()

    @staticmethod
    def _patch_file(file_cwd, patch_cwd):
        cmd = [
            'patch',
            file_cwd,
            patch_cwd,
        ]
        logging.info('Running patch command: {}'.format(cmd))
        subprocess.Popen(cmd)

    def _patch_archon_renderer(self):
        logging.info('Start patching files')
        archon_renderer_path = os.path.join(
            self._pcode_unpacked_path,
            'repo/node_modules/@yandex-int/archon-renderer/components/renderer.js'
        )
        with open("archon-renderer.patch", "wb") as f:
            f.write(patch_renderer.ARCHON_RENDERER_PATCH)
        self._patch_file(archon_renderer_path, "archon-renderer.patch")
        load_pcode_resource_path = os.path.join(
            self._pcode_unpacked_path,
            'repo/.config/archon/components/load-pcode-resource.js'
        )
        with open("load-pcode-resource.patch", "wb") as f:
            f.write(patch_renderer.LOAD_PCODE_RESOURCE_PATCH)
        self._patch_file(load_pcode_resource_path, "load-pcode-resource.patch")
        pcode_renderer_path = os.path.join(
            self._pcode_unpacked_path,
            'repo/.config/archon/commands/pcode-renderer.js'
        )
        with open("pcode-renderer.patch", "wb") as f:
            f.write(patch_renderer.PCODE_RENDERER_PATCH)
        self._patch_file(pcode_renderer_path, "pcode-renderer.patch")
        logging.info('End patching files')

    def _run_renderer(self):
        os.environ['PATH'] += os.pathsep + os.path.join(self._pcode_unpacked_path, self.adapter.get_node_relpath())

        root_dir = os.getcwd()
        os.chdir(os.path.join(self._pcode_unpacked_path, 'repo'))
        cmd = [
            'npx',
            'archon',
            'pcode-renderer',
            '--rendererPort',
            str(self._ports['renderer_port']),
            '--renderer-devops-port',
            str(self._ports['renderer_admin_port']),
            '--pcode-resource-id',
            str(self.adapter.get_web_pcode_micro_package_id()),
            '--pcode-loader-resource-id',
            str(self.adapter.get_pcode_loader_superbundle_id()),
            '--pcode-vas-resource-id',
            str(self.adapter.get_pcode_vas_superbundle_id()),
        ]
        logging.info('Running renderer with command: {}'.format(cmd))

        self._renderer_process_log_context = sdk2.helpers.ProcessLog(self.adapter.task_instance, 'renderer.run')
        self._renderer_process_log_context.__enter__()
        self._renderer_process = subprocess.Popen(
            cmd, stdout=self._renderer_process_log_context.stdout, stderr=self._renderer_process_log_context.stderr
        )
        os.chdir(root_dir)

    def _stop_renderer(self, *args):
        self._renderer_process.kill()
        self._renderer_process.communicate()
        self._renderer_process_log_context.__exit__(*args)
        self._renderer_process_log_context = None

    def __enter__(self):
        self._cachedaemon.__enter__()
        self._run_renderer()

        self._active = True
        wait_ping("http://localhost:{}/admin?action=ping".format(self._ports['renderer_port']), max_pause=60)

        return self

    def __exit__(self, *args):
        self._stop_renderer(*args)
        self._cachedaemon.__exit__(*args)
        self._active = False

    def get_port(self):
        if self._active:
            return self._ports["renderer_port"]
        else:
            return None
