import json
import base64
import xml.dom.minidom as md


class BaseContentParser(object):

    def __init__(
        self,
        content,
        is_base64=False,
    ):
        self.content = content
        self.content_type = "UNKNOWN"
        if is_base64 \
                and len(self.content.splitlines()) == 1 \
                and len(self.content) > 50 \
                and ' ' not in self.content:
            self.content = self._decode_base64()

        self.is_json = False
        self.is_xml = False
        self.is_html = False

        self.content = self._try_load_json()
        self.content = self._try_load_xml()
        self.content = self._try_load_html()

        assert self.is_json + self.is_xml + self.is_html <= 1

    def empty_content(self):
        return len(self.content) == 0

    def serialized_content(self):
        if self.is_json:
            return self._serialized_json()
        if self.is_xml:
            return self._serialized_xml()
        if self.is_html:
            return self._serialized_html()
        return self.content.splitlines()

    def _decode_base64(self):
        if not len(self.content):
            return self.content
        return base64.b64decode(self.content)

    def _try_load_json(self):
        result = self.content
        try:
            parsed_json = json.loads(self.content)
            self.is_json = isinstance(parsed_json, dict)
            if self.is_json:
                result = parsed_json
                self.content_type = "JSON"
        except ValueError:
            pass
        return result

    def _serialized_json(self):
        return json.dumps(self.content, sort_keys=True, indent=4).splitlines()

    def _serialized_xml(self):
        return self.content.toprettyxml(indent=' ').splitlines()

    def _serialized_html(self):
        return self.content.splitlines()

    def _try_load_xml(self):
        result = self.content
        try:
            result = md.parseString(self.content)
            self.is_xml = True
            self.content_type = "XML"
        except:
            pass
        return result

    def _try_load_html(self):
        try:
            if self.content.startswith("<!DOCTYPE html>"):
                self.is_html = True
                self.content_type = "HTML"
        except:
            pass
        return self.content


class JanpuContentParser(BaseContentParser):

    def __init__(self, content):
        super(JanpuContentParser, self).__init__(content, is_base64=True)
