from abc import ABCMeta, abstractmethod
import json
import base64
import xml.dom.minidom as md

from sandbox import sdk2


class BaseModule(object):

    __metaclass__ = ABCMeta

    @abstractmethod
    def __init__(self, parameters):
        self.name = None

    @staticmethod
    @abstractmethod
    def get_init_parameters_class():
        pass

    @abstractmethod
    def process_headers(self, headers):
        pass

    @abstractmethod
    def process_content(self, content, content_type):
        pass


class Str2JsonParseModule(BaseModule):

    def __init__(self, parameters):
        super(Str2JsonParseModule, self).__init__(parameters)
        self.name = "Str2JsonParseModule"

    @staticmethod
    def get_init_parameters_class():
        return None

    def process_headers(self, headers):
        return headers

    def process_content(self, content, content_type):
        if not isinstance(content, str):
            return content, content_type
        result = content
        new_content_type = content_type
        try:
            parsed_json = json.loads(content)
            is_json = isinstance(parsed_json, dict)
            if is_json:
                result = parsed_json
                new_content_type = "JSON"
        except ValueError:
            pass
        return result, new_content_type


class Json2StrParseModule(BaseModule):

    def __init__(self, parameters):
        super(Json2StrParseModule, self).__init__(parameters)
        self.name = "Json2StrParseModule"

    @staticmethod
    def get_init_parameters_class():
        return None

    def process_headers(self, headers):
        return headers

    def process_content(self, content, content_type):
        if not isinstance(content, dict):
            return content, content_type

        return json.dumps(content, sort_keys=True, indent=2), "JSON"


class RemoveJsonKeysModule(BaseModule):

    def __init__(self, parameters):
        super(RemoveJsonKeysModule, self).__init__(parameters)
        self.json_keys_to_delete = parameters.json_keys_to_delete
        self.name = "RemoveJsonKeysModule"

    @staticmethod
    def get_init_parameters_class():
        class Parameters(sdk2.Task.Parameters):
            json_keys_to_delete = sdk2.parameters.List(
                "List of keys to be deleted from json during comparison",
                default=[]
            )
        return Parameters

    def process_headers(self, headers):
        return headers

    def _remove_keys_from_dict(self, cur_json):
        if isinstance(cur_json, dict):
            keys = list(cur_json.keys())
            for key in keys:
                if key in self.json_keys_to_delete:
                    del cur_json[key]
                else:
                    self._remove_keys_from_dict(cur_json[key])
        elif isinstance(cur_json, list):
            for value in cur_json:
                self._remove_keys_from_dict(value)
        return cur_json

    def process_content(self, content, content_type):
        if not isinstance(content, dict):
            return content, content_type
        return self._remove_keys_from_dict(content), "JSON"


class DecodeBase64Module(BaseModule):

    def __init__(self, parameters):
        super(DecodeBase64Module, self).__init__(parameters)
        self.name = "DecodeBase64Module"

    @staticmethod
    def get_init_parameters_class():
        return None

    def process_headers(self, headers):
        return headers

    def process_content(self, content, content_type):
        if len(content.splitlines()) == 1 and len(content) > 50 and ' ' not in content:
            return base64.b64decode(content), content_type
        return content, content_type


class Str2XmlParseModule(BaseModule):
    def __init__(self, parameters):
        super(Str2XmlParseModule, self).__init__(parameters)
        self.name = "Str2XmlParseModule"

    @staticmethod
    def get_init_parameters_class():
        return None

    def process_headers(self, headers):
        return headers

    def process_content(self, content, content_type):
        if content_type != "UNKNOWN":
            return content, content_type

        new_content = content
        new_content_type = content_type
        try:
            new_content = md.parseString(content)
            new_content_type = "XML"
        except:
            pass
        return new_content, new_content_type


class Xml2StrParseModule(BaseModule):
    def __init__(self, parameters):
        super(Xml2StrParseModule, self).__init__(parameters)
        self.name = "Xml2StrParseModule"

    @staticmethod
    def get_init_parameters_class():
        return None

    def process_headers(self, headers):
        return headers

    def process_content(self, content, content_type):
        if content_type != "XML":
            return content, content_type
        return content.toprettyxml(indent=' '), content_type


class HeadersPaintModule(BaseModule):
    def __init__(self, parameters):
        super(HeadersPaintModule, self).__init__(parameters)
        self.headers_to_replace = parameters.headers_to_replace

    @staticmethod
    def get_init_parameters_class():
        class Parameters(sdk2.Task.Parameters):
            headers_to_replace = sdk2.parameters.List(
                "Headers painting",
                default=["Date", "Content-Length", "X-Yandex-Req-Id"]
            )
        return Parameters

    def process_headers(self, headers):
        for header in self.headers_to_replace:
            headers[header] = "\"<PAINTED>\""
        return headers

    def process_content(self, content, content_type):
        return content, content_type
