import os
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
from sandbox.sdk2.helpers import subprocess as sp

from sandbox import sdk2
from sandbox.projects.pcode.qa.tasks.JanpuGraphSimpleShootTask import JanpuGraphSimpleShootTask
from sandbox.projects.pcode.qa.tasks.PcodePrepareAmmoResources import PcodePrepareAmmoResources
from sandbox.projects.pcode.qa.resource_types import PcodeJanpuPerfAmmoGz
from sandbox.projects.pcode.resources import PcodeGeneratePerfAmmoBin, PcodePreparePhantomRequestsBin


JDK_RESOURCE = 1901452000
MOBILE_ADS_SDK_RESOURCE = 3255811435
YT_UPLOADER_RESOURCE = 890323477


class PcodeBuildPerfAmmo(PcodePrepareAmmoResources):

    class Parameters(PcodePrepareAmmoResources.Parameters):
        nodes_to_grep_context = sdk2.parameters.List("nodes_to_grep_context",
                                                     default=["SDK_ROUTER", "SDK_RENDERER"])
        ya_token = sdk2.parameters.YavSecretWithKey(
            'YA token',
            required=True,
        )
        yt_token = sdk2.parameters.YavSecretWithKey(
            'YT token',
            required=True,
        )
        sandbox_token = sdk2.parameters.YavSecretWithKey(
            'SANDBOX token',
            required=True,
        )
        max_request_id = sdk2.parameters.Integer(
            'Max RequestID in grep contexts',
            required=True, default=30000
        )

        generate_perf_ammo_bin = sdk2.parameters.LastReleasedResource(
            'PCODE_GENERATE_PERF_AMMO_BIN resource',
            resource_type=PcodeGeneratePerfAmmoBin,
        )
        prepare_phantom_requests_bin = sdk2.parameters.LastReleasedResource(
            'PCODE_PREPARE_PHANTOM_REQUESTS_BIN resource',
            resource_type=PcodePreparePhantomRequestsBin,
        )

        with sdk2.parameters.Output:
            output_yt_path = sdk2.parameters.String("Test param")
            perf_ammo_resource = sdk2.parameters.Resource('Resource with perf ammo')

    def on_execute(self):
        yt_token = self.Parameters.yt_token.data()[self.Parameters.yt_token.default_key]

        with self.memoize_stage['Prepare ammo resources']:
            super(PcodeBuildPerfAmmo, self).on_execute()

        with self.memoize_stage['Running shoot task'](commit_on_entrance=False):
            child = self._spawn_shoot_task()
            child.Requirements.cores = 8
            child.Requirements.ram = 16 * 1024
            child.Requirements.client_tags = ctc.Tag.Group.LINUX & ctc.Tag.SSD
            child.save()
            child.enqueue()
            self.Context.shoot_task_id = child.id
            raise sdk2.WaitTask(child, (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))

        child = self.find(JanpuGraphSimpleShootTask, id=self.Context.shoot_task_id).first()
        context_yt_path = child.Parameters.output_yt_path
        output_file = context_yt_path.split('/')[-1]

        with self.memoize_stage['Running generate_perf_ammo'](commit_on_entrance=False):
            env = os.environ.copy()
            env['YT_TOKEN'] = yt_token
            generate_perf_ammo_bin_res = sdk2.ResourceData(self.Parameters.generate_perf_ammo_bin)
            cmd = [
                str(generate_perf_ammo_bin_res.path),
                "-i",
                context_yt_path,
                "-o",
                output_file,
                "-n",
                str(self.Parameters.max_request_id)
            ]
            with sdk2.helpers.ProcessLog(self, logger="generate_perf_ammo_log") as log:
                sp.check_call(cmd, stdout=log.stdout, stderr=log.stderr, env=env)

        phantom_output_file = output_file + "_phantom"
        with self.memoize_stage['Running prepare_phantom_format'](commit_on_entrance=False):
            prepare_phantom_requests_bin_res = sdk2.ResourceData(self.Parameters.prepare_phantom_requests_bin)
            cmd = [
                str(prepare_phantom_requests_bin_res.path),
                "--address",
                "localhost:10050",
                "--graph-name",
                self.Parameters.graph_name.split("__")[-1],
                "-f",
                output_file,
                "--split-pattern",
                "splittedJson="
            ]
            with sdk2.helpers.ProcessLog(self, logger="prepare_phantom_format_log") as log, open(phantom_output_file, 'wb') as fp:
                sp.check_call(cmd, stdout=fp, stderr=log.stderr)

        sp.check_call(["gzip", phantom_output_file])
        self.Parameters.perf_ammo_resource = sdk2.ResourceData(PcodeJanpuPerfAmmoGz(
            self,
            'for perf testing',
            os.path.abspath(phantom_output_file + '.gz'),
        )).ready()

    def _spawn_shoot_task(self):
        return JanpuGraphSimpleShootTask(
            self,
            requestlog_resource=self.Parameters.requestlog_resource.id,
            cache_daemon_stub_resource=self.Parameters.cache_daemon_stub_data.id,
            ya_token=self.Parameters.ya_token,
            yt_token=self.Parameters.yt_token,
            sandbox_token=self.Parameters.sandbox_token,
            description="Child of {}".format(self.id),
            owner=self.owner,
            core_count=16,
            queryargs_update_dict={
                "json_dump_format": "context",
                "srcrwr": [
                    "SDK_ROUTER:localhost:10050",
                    "SDK_DEMO:localhost:10050",
                    "SDK_RENDERER:localhost:10050",
                    "SDK_BLOCK_ID_ROUTER:localhost:10050",
                    "AD_ENGINE:localhost:44751",
                    "AD_ENGINE_GET_BLOCK_ID:localhost:44444",
                    "AD_PROMO:localhost:45678"
                ],
                "json_dump_requests": self.Parameters.nodes_to_grep_context
            },
            headers_update_dict={
                "x-yabs-test-random": "42",
                "x-yabs-test-time": "996029623",
                "accept-encoding": "",
                "X-Yandex-Internal-Request": 1
            },
            cache_daemon_ports_map={
                "AD_ENGINE_GET_BLOCK_ID": 44444,
                "AD_ENGINE": 44751,
                "AD_PROMO": 45678,
                "LAAS": 39981,
                "ADFOX": 46183
            },
            jdk_resource=JDK_RESOURCE,
            mobile_ads_sdk_port=10050,
            mobile_ads_sdk_resource=MOBILE_ADS_SDK_RESOURCE,
            port=10094,
            request_timeout=10,
            shoot_request_limit=200000,
            test_id_header="X-Yabs-Nanpu-Req-Id",
            test_id_queryarg="nanpu-request-id",
            vertical="PCODE",
            yt_table_prefix="//home/pcode/shoot_results",
            yt_token_vault_key="PCODE_YT_TOKEN",
            yt_ttl_days=3,
            yt_upload=True,
            yt_uploader_resource=YT_UPLOADER_RESOURCE,
            yt_uploader_resource_type="plain"
        )
