import os
import subprocess

from sandbox import sdk2
from sandbox.common.types.misc import DnsType
from sandbox.projects.yabs.ssr.util import run_command

from sandbox.projects.pcode.qa.resource_types import PcodeRendererResource


NVM_URL = 'https://raw.githubusercontent.com/nvm-sh/nvm/v0.37.2/install.sh'
CONTAINER_RESOURCE = 2090878873


class PcodePrepareRendererTask(sdk2.Task):

    class Requirements(sdk2.Requirements):
        container_resource = CONTAINER_RESOURCE
        dns = DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        description = "Creating PcodeRendererResource"

        working_dir = sdk2.parameters.String(
            'Name for directory with code',
            default='renderer',
        )

        with sdk2.parameters.Group('Authentification parameters') as repo_params:
            repo_url = sdk2.parameters.String(
                'PCODE repo url',
                default='https://github.yandex-team.ru/wanya/pcode.git',
            )
            repo_branch = sdk2.parameters.String(
                'Branch name to be cloned',
                default='master',
            )
            repo_commit = sdk2.parameters.String('Commit branch will be reset on')

        with sdk2.parameters.Group('Authentification parameters') as authentification_params:
            user_email = sdk2.parameters.String(
                'User email used for authentification',
                default='aydarboss@yandex-team.ru',
            )
            user_name = sdk2.parameters.String(
                'User name used for authentification',
                default='Aydarbek Khusainov',
            )
            ya_token = sdk2.parameters.YavSecretWithKey("YA token", required=True)

        with sdk2.parameters.Output:
            renderer_resource = sdk2.parameters.Resource('Resource with Pcode Report-Renderer Resource data',
                                                         resource_type=PcodeRendererResource)

    def _load_code(self):
        def run_command_with_stderr(command):
            return run_command(command, stderr=subprocess.STDOUT)

        root_dir = os.getcwd()
        working_dir = str(self.Parameters.working_dir)
        os.makedirs(working_dir)
        os.chdir(working_dir)

        working_dir = os.getcwd()
        os.makedirs('.nvm')
        os.environ['NVM_DIR'] = os.path.join(working_dir, '.nvm')

        run_command_with_stderr(
            'git config --global user.email "{user_email}"'.format(
                user_email=self.Parameters.user_email,
            )
        )
        run_command_with_stderr(
            'git config --global user.name "{user_name}"'.format(
                user_name=self.Parameters.user_name,
            )
        )
        run_command_with_stderr(
            'git clone {repo_url} -b {repo_branch} repo'.format(
                repo_url=self.Parameters.repo_url,
                repo_branch=self.Parameters.repo_branch,
            )
        )
        os.chdir('repo')
        if self.Parameters.repo_commit:
            run_command(
                'git reset --hard {repo_commit}'.format(
                    repo_commit=self.Parameters.repo_commit,
                )
            )

        with open('.nvmrc', 'r') as f:
            version = f.readline().strip(' \n')

        run_command_with_stderr(
            'curl -o- "{nvm_url}" | bash && '
            'source "$NVM_DIR/nvm.sh" || true && '
            'nvm install {version}'.format(
                nvm_url=NVM_URL,
                version=version,
            )
        )

        node_path = os.path.join(os.environ['NVM_DIR'], 'versions', 'node', 'v' + version, 'bin')
        os.environ['PATH'] += os.pathsep + node_path

        run_command_with_stderr('npm config set registry https://npm.yandex-team.ru')
        run_command_with_stderr(
            'npm config set email "{user_email}"'.format(
                user_email=self.Parameters.user_email,
            )
        )
        run_command_with_stderr(
            'npm config set _auth {token}'.format(
                token=self.Parameters.ya_token.data()[self.Parameters.ya_token.default_key]
            )
        )

        run_command_with_stderr('npm install yarn')
        run_command_with_stderr('npm install @yandex-int/archon @yandex-int/archon-renderer')
        run_command_with_stderr('npx yarn || true')

        os.chdir(root_dir)

        return os.path.relpath(node_path, working_dir)

    def on_execute(self):
        renderer_resource = PcodeRendererResource(
            self,
            'Pcode Report-Renderer Resource',
            self.Parameters.working_dir,
            repo_url=self.Parameters.repo_url,
            repo_branch=self.Parameters.repo_branch,
            repo_commit=self.Parameters.repo_commit,
        )
        renderer_resource.node_relpath = self._load_code()
        self.Parameters.renderer_resource = renderer_resource
