import os
import json
import logging
import posixpath
import textwrap
from multiprocessing import Pool, cpu_count
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.projects.yabs.qa.resource_types import YABS_REPORT_RESOURCE
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.report import CmpReport
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp import YabsServerB2BFuncShootCmp
from sandbox.projects.yabs.qa.utils.resource import sync_resource

from sandbox.projects.pcode.qa.resource_types import PcodeRendererDiffReport, PcodeRendererDiffResult
from sandbox.projects.pcode.qa.response_differ.compare_data_reducer import CompareDataReducer
from sandbox.projects.pcode.qa.response_differ import modules
from sandbox.projects.pcode.qa.response_differ.utils import unpack_request_log
from sandbox.projects.pcode.qa.tasks.JanpuDiffTask.processing import process_results

from parameters import DiffTaskParameters
from response_parse_mapper import ResponseParseMapper

REPORT_DIR = 'report_dir'

# Order important
MODULES = [
    modules.Str2JsonParseModule,
    modules.RemoveJsonKeysModule,
    modules.Json2StrParseModule,
]


class PcodeRendererDiffTask(sdk2.Task):
    diff_results = []
    pre_codes = set()
    test_codes = set()
    handlers = set()
    tags = set()

    Parameters = DiffTaskParameters

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    class Context(sdk2.Task.Context):
        has_diff = True
        num_of_requests_with_diff = 0
        num_of_requests = 0

    def on_execute(self):
        import yt.wrapper as yt
        yt.config['proxy']['url'] = self.Parameters.yt_cluster
        yt.config['token'] = self._get_yt_token()
        self._prepare_report_dir()
        self.yt = yt
        results_directory = self._get_results_directory()
        diff_path = yt.ypath_join(results_directory, '{task_id}_pcode_renderer_shoot_diff'.format(task_id=self.id))
        logging.info('Diff path: {}'.format(diff_path))

        preshoot_result_path = self.Parameters.preshoot_result.yt_table_path
        patched_shoot_result_path = self.Parameters.patched_shoot_result.yt_table_path
        parse_mapper_instance = self._init_response_parse_mapper_instance()

        def read_data(yt_path):
            logging.info('Reading shoot results from path: {}'.format(yt_path))
            return self._read_yt_table(yt_path, 'RequestID', ['HttpCode', 'Url', 'Data'])

        pre_data = read_data(preshoot_result_path)
        test_data = read_data(patched_shoot_result_path)

        pre_mapped_data = dict(filter(None, self._run_map(parse_mapper_instance, pre_data.iteritems())))
        test_mapped_data = dict(filter(None, self._run_map(parse_mapper_instance, test_data.iteritems())))
        diff_data = self._comparing_results(pre_mapped_data, test_mapped_data, diff_path)

        logging.info('Setting expiration time for result table.')
        expiration_time = datetime.utcnow() + timedelta(days=self.Parameters.yt_ttl)
        self.yt.set(diff_path + '/@expiration_time', expiration_time.isoformat() + 'Z')
        logging.info('Set expiration time for result table to {}.'.format(expiration_time.isoformat()))
        self.Parameters.diff_table_path = diff_path

        self._create_report(pre_data, test_data, diff_data, diff_path)

        if self.Parameters.fail_task_if_has_diff and self.Context.has_diff:
            raise Exception("Compare has diff")

    def _comparing_results(self, pre_data, test_data, diff_path):
        logging.info('Comparing results')
        pre_request_ids, test_request_ids = set(pre_data.keys()), set(test_data.keys())
        intersec_request_ids = pre_request_ids & test_request_ids
        diff_request_ids = pre_request_ids ^ test_request_ids
        if diff_request_ids:
            logging.info('ATTENTION!!! Missing test ids in {}: {}'.format(
                'test' if len(pre_request_ids) > len(test_request_ids) else 'pre',
                list(diff_request_ids)
            ))

        compare_function_args = (
            (request_id,
             pre_data[request_id],
             test_data[request_id])
            for request_id in intersec_request_ids
        )
        results = dict(filter(None, self._run_map(CompareDataReducer(), compare_function_args)))
        logging.info('Finished comparison.')
        logging.info('Saving results to YT...')
        self.yt.write_table(self.yt.TablePath(diff_path), results.values())
        return results

    def _create_report(self, pre_data, test_data, diff_data, diff_table_path):
        logging.info('Unpaccking request log.')
        requests = unpack_request_log(self.Parameters.requestlog_resource, r'x-yabs-ssr-req-id: (\d*)')
        bad_requests_ids = self.Parameters.bad_requests_ids
        n_jobs = self.Parameters.n_jobs

        logging.info('Processing results.')
        results = process_results(requests, pre_data, test_data, diff_data, bad_requests_ids, n_jobs, REPORT_DIR)

        self.Context.num_of_requests = len(results)

        for result in results:
            if result['has_diff']:
                self.Context.num_of_requests_with_diff += 1

            self.pre_codes.add(result['pre_code'])
            self.test_codes.add(result['test_code'])
            self.handlers.add(result['handler'])
            self.tags.update(result['tags'])

            self.diff_results.append(
                {
                    'status': ('failed' if result['has_diff'] else 'passed'),
                    'search': {
                        'pre.code': result['pre_code'],
                        'test.code': result['test_code'],
                        'handler': result['handler'],
                        'tags': list(result['tags']),
                    },
                    'name': str(int(result['test_id'])),
                    'id': int(result['test_id']),
                    'diffLinesCount': 20,
                }
            )

        metadata = {
            "Tests": len(self.diff_results),
            "Failures": self.Context.num_of_requests_with_diff,
        }

        report = {
            'search': {
                'pre.code': list(self.pre_codes),
                'test.code': list(self.test_codes),
                'handler': list(self.handlers),
                'tags': list(self.tags),
            },
            'meta': [{'title': title, 'value': value} for title, value in metadata.iteritems()],
            'results': self.diff_results,
        }

        report_file_path = os.path.join(REPORT_DIR, 'report.json')
        with open(report_file_path, 'w') as report_file:
            json.dump(report, report_file)

        logging.info(
            'Finished. {} out of {} requests has diff.'.format(
                self.Context.num_of_requests_with_diff, self.Context.num_of_requests
            )
        )

        self.Context.has_diff = bool(self.Context.num_of_requests_with_diff)

        report_resource = PcodeRendererDiffReport(self, 'Report resource', REPORT_DIR, ttl=self.Parameters.ttl)
        sdk2.ResourceData(report_resource).ready()

        report_url = YabsServerB2BFuncShootCmp.get_resource_url(self.id, REPORT_DIR, report_resource.id)
        diff_url = posixpath.join(report_url, 'index.html')

        execution_report = textwrap.dedent(
            '''\
            {failures} out of {total_tests} tests failed.
            <a href="{diff_url}" target="_blank">Diff viewer</a>
            <a href='{diff_table_url}' target='_blank'>Diff table</a>
        '''
        ).format(
            failures=self.Context.num_of_requests_with_diff,
            total_tests=self.Context.num_of_requests,
            diff_url=diff_url,
            diff_table_url='https://yt.yandex-team.ru/hahn/navigation?path={}'.format(diff_table_path),
        )

        self.set_info(execution_report, do_escape=False)

        with open('report.html', 'w') as file:
            file.write(execution_report.replace('\n', '<br>'))

        self.Parameters.report = PcodeRendererDiffResult(self, 'Report resource', 'report.html', ttl=self.Parameters.ttl)
        sdk2.ResourceData(self.Parameters.report).ready()
        logging.info('Created resource with YT path.')

    def _read_yt_table(self, table_path, key_column, columns=None):
        columns_to_read = [key_column] + columns if columns is not None else None
        result = {}
        for row in self.yt.read_table(self.yt.TablePath(table_path, columns=columns_to_read)):
            key_value = row[key_column]
            columns_value = {col: row[col] for col in columns} if columns is not None else row
            result[key_value] = columns_value
        return result

    def _init_response_parse_mapper_instance(self):
        inited_modules = []
        for module in MODULES:
            inited_modules.append(module(self.Parameters))
        return ResponseParseMapper(inited_modules)

    def _run_map(self, map_function, items):
        n_jobs = self.Parameters.n_jobs
        if n_jobs == 1:
            results = map(map_function, items)
        else:
            n_jobs = n_jobs if n_jobs is not None else int(cpu_count() * 1.5)
            pool = Pool(n_jobs)
            results = pool.map(map_function, items, 128)
            pool.close()
            pool.join()
        return results

    def _get_results_directory(self):
        results_directory = self.Parameters.diff_table_prefix_path
        if not self.yt.exists(results_directory):
            logging.info('Creating directory for results in YT: {}'.format(results_directory))
            self.yt.mkdir(results_directory, recursive=True)
        return results_directory

    def _prepare_report_dir(self):
        if not os.path.exists(REPORT_DIR):
            os.mkdir(REPORT_DIR)
        template_path = sync_resource(resource=self.Parameters.template_resource, resource_type=YABS_REPORT_RESOURCE)
        CmpReport(REPORT_DIR).prepare(template_path)

    def _get_yt_token(self):
        return self.Parameters.yt_token.data()[self.Parameters.yt_token.default_key]
