import os
import logging

import yaml

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import resource as ctr
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common import binary_task

from sandbox.projects.pcode.zfp.utils import validate_params

from solomon import update_alert
from juggler import update_aggregate


class PcodeZfpMonitoringAutoupdate(binary_task.LastBinaryTaskRelease, sdk2.Task):
    name = "PCODE_ZFP_MONITORING_AUTOUPDATE"

    class Parameters(sdk2.Task.Parameters):

        base_path = sdk2.parameters.String('Path to solomon alerts and juggler aggregates', default_value="/adv/pcode/zfp/alert_configs/pcode_banner", required=True)
        template_path = sdk2.parameters.String('Path to templates solomon programs', default_value="/adv/pcode/zfp/solomon_program_templates", required=True)
        alerts = sdk2.parameters.String('List of alerts to update', default_value="")
        update_solomon_alerts = sdk2.parameters.Bool('Update solomon alerts', default_value=True)
        use_solomon_prestable = sdk2.parameters.Bool('Use solomon-prestable', default_value=False)
        update_juggler_aggregates = sdk2.parameters.Bool('Update juggler aggregates', default_value=True)
        solomon_token = sdk2.parameters.YavSecretWithKey(
            "Solomon token",
            default="sec-01cv39zerwcz9zk1wjxght1zb8#solomon-oauth-token",

        )
        juggler_token = sdk2.parameters.YavSecretWithKey(
            "Juggler token",
            default="sec-01cv39zerwcz9zk1wjxght1zb8#juggler-oauth-token",
        )

        ext_params = binary_task.binary_release_parameters(stable=True)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": PcodeZfpMonitoringAutoupdate.name, "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def on_execute(self):

        os.chdir(str(self.path()))
        Arcadia.export(Arcadia.trunk_url(self.Parameters.base_path), str(self.path('base_path')))
        Arcadia.export(Arcadia.trunk_url(self.Parameters.template_path), str(self.path('template_path')))
        solomon_token = self.Parameters.solomon_token.data()[self.Parameters.solomon_token.default_key]
        juggler_token = self.Parameters.juggler_token.data()[self.Parameters.juggler_token.default_key]

        if self.Parameters.alerts:
            for alert in self.Parameters.alerts.split(','):
                self.update_alert(alert, solomon_token, juggler_token)

    def update_alert(self, alert, solomon_token, juggler_token):
        logging.info("Update alert: {}".format(alert))
        base_path = str(self.path('base_path'))
        path_to_params = os.path.join(base_path, alert, 'params.yaml')
        with open(path_to_params) as params_file:
            params = yaml.safe_load(params_file.read())
        # validate params
        errors = validate_params(params, alert, self.Parameters.update_solomon_alerts, self.Parameters.update_juggler_aggregates)

        if errors:
            raise TaskFailure('\n'.join(errors))

        common_settings = params["common_settings"]
        alert_settings = params["alert_settings"]
        for alert_name, settings in alert_settings.items():
            aggregate_settings = settings["aggregate"]
            if self.Parameters.update_solomon_alerts:
                update_alert(alert_name, common_settings, aggregate_settings, settings["solomon"], str(self.path('template_path')), solomon_token, use_prestable=self.Parameters.use_solomon_prestable)

            if self.Parameters.update_juggler_aggregates:
                update_aggregate(alert_name, common_settings, settings, juggler_token)

        logging.info("Update alert success: {}".format(alert))
