import logging
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task

from sandbox.projects.pcode.zfp.resource_types import PcodeZfpPackage


TEMPLATE_WORKFLOW_ID = '3a12584c-5b3b-4a91-8af4-c6d14c22173d'
TARGET_WORKFLOW_ID = 'b85a9d1c-9ab7-49a1-b284-ae46b03b6412'

ARCADIA_PATH_TEMPLATE = 'antiadblock/tasks/monitoring_update/programs/{}/{}'

INSTANCE_TEMPLATE_URL = 'https://nirvana.yandex-team.ru/flow/{}/{}/graph'


class PcodeZfpRunCalculation(binary_task.LastBinaryTaskRelease, sdk2.Task):
    name = "PCODE_ZFP_RUN_CALCULATION"

    class Parameters(sdk2.Task.Parameters):
        zfp_package_resource = sdk2.parameters.Resource(
            'Package with alert programs and aggregate configs for ZFP test running',
            resource_type=PcodeZfpPackage)

        alert_ids = sdk2.parameters.List('alert ids to run calculation', required=True)
        days = sdk2.parameters.Integer('count days for calculation', default_value=45, required=True)
        nirvana_token = sdk2.parameters.YavSecretWithKey(
            "Nirvana OAUTH TOKEN",
            default="sec-01cv39zerwcz9zk1wjxght1zb8#nirvana-oauth-token",

        )
        startrek_queue_name = sdk2.parameters.String('Startrek queue name', default_value='PCODEALERTS', required=True)
        comment_pr = sdk2.parameters.Bool('Push comment to pull request', default=False)
        with comment_pr.value[True]:
            pr_id = sdk2.parameters.String('Pull request id', default_value="")
            arcanum_token = sdk2.parameters.YavSecretWithKey(
                "Arcanum token",
                default="sec-01cv39zerwcz9zk1wjxght1zb8#arcanum-token",

            )

        trace_instances = sdk2.parameters.Bool("Trace instances", default=True)
        with trace_instances.value[True]:
            retry_period = sdk2.parameters.Integer(
                "Time period to check request status (in seconds)",
                default=60 * 5
            )

        debug_mode = sdk2.parameters.Bool("Debug mode", default=False)
        with debug_mode.value[True]:
            period_from = sdk2.parameters.String("Start datetime in format %Y-%m-%d %H:%M:%S", default_value="")
            period_to = sdk2.parameters.String("Finish datetime in format %Y-%m-%d %H:%M:%S", default_value="")

        with sdk2.parameters.Output:
            instance_url = sdk2.parameters.String('Nirvana instance url', default="")
            is_finished = sdk2.parameters.Bool('Nirvana instances is finished ', default=False)

        ext_params = binary_task.binary_release_parameters(stable=True)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": PcodeZfpRunCalculation.name, "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    class Context(sdk2.Task.Context):
        is_finished = {}

    def on_execute(self):
        from adv.pcode.zfp.utils.arcanum import publish_pr_comment

        with self.memoize_stage.send_request:
            instance_id = self._create_pipeline_instance()
            self.Parameters.instance_url = INSTANCE_TEMPLATE_URL.format(TARGET_WORKFLOW_ID, instance_id)
            self.Context.is_finished[instance_id] = False
            # push comment to pr
            if self.Parameters.comment_pr and self.Parameters.pr_id:
                arcanum_token = self.Parameters.arcanum_token.data()[self.Parameters.arcanum_token.default_key]
                comment = "Start eval graph for {}.\nNirvana url {}".format(",".join(self.Parameters.alert_ids),  self.Parameters.instance_url)
                publish_pr_comment(comment, self.Parameters.pr_id, arcanum_token)

        if self.Parameters.trace_instances:
            workflow_execution_states = self._get_instances_status()
            for state in workflow_execution_states:
                logging.info(state)

            if any(state['status'] != 'completed' for state in workflow_execution_states):
                self._sleep()

            if any(state['result'] != 'success' for state in workflow_execution_states):
                raise TaskFailure("Eval graphs didn't complete successfully")

            self.Parameters.is_finished = True

    def _sleep(self):
        raise sdk2.WaitTime(self.Parameters.retry_period)

    def _get_instances_status(self):
        import nirvana_api
        nirvana_token = self.Parameters.nirvana_token.data()[self.Parameters.nirvana_token.default_key]
        api = nirvana_api.NirvanaApi(oauth_token=nirvana_token, ssl_verify=False)
        return api.get_workflow_execution_states(self.Context.is_finished.keys())

    def _create_pipeline_instance(self):
        import nirvana_api
        nirvana_token = self.Parameters.nirvana_token.data()[self.Parameters.nirvana_token.default_key]
        api = nirvana_api.NirvanaApi(oauth_token=nirvana_token, ssl_verify=False)
        template_instance_id = api.find_workflows(pattern=TEMPLATE_WORKFLOW_ID)[0].instanceId

        target_instance_id = api.clone_workflow_instance(
            workflow_id=TEMPLATE_WORKFLOW_ID,
            workflow_instance_id=template_instance_id,
            target_workflow_id=TARGET_WORKFLOW_ID,
        )
        if not self.Parameters.debug_mode:
            period_to = datetime.now()
            period_from = period_to - timedelta(days=int(self.Parameters.days))
        else:
            period_to = datetime.strptime(self.Parameters.period_to, "%Y-%m-%d %H:%M:%S")
            period_from = datetime.strptime(self.Parameters.period_from, "%Y-%m-%d %H:%M:%S")
        param_values = [nirvana_api.ParameterValue('alert_ids', self.Parameters.alert_ids),
                        nirvana_api.ParameterValue('period_to', period_to.strftime('%s')),
                        nirvana_api.ParameterValue('period_from', period_from.strftime('%s')),
                        nirvana_api.ParameterValue('startrek_queue_name', self.Parameters.startrek_queue_name),
                        nirvana_api.ParameterValue('zfp_package_resource_id', str(self.Parameters.zfp_package_resource.id))]

        if self.Parameters.comment_pr and self.Parameters.pr_id:
            param_values.append(nirvana_api.ParameterValue('pull_request_id', self.Parameters.pr_id))

        api.set_global_parameters(
            workflow_id=TARGET_WORKFLOW_ID,
            workflow_instance_id=target_instance_id,
            param_values=param_values,
        )

        instance_id = api.start_workflow(
            workflow_id=TARGET_WORKFLOW_ID,
            workflow_instance_id=target_instance_id,
        )
        return instance_id
