import os
import logging
import tarfile

import yaml

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task

from sandbox.projects.pcode.zfp.resource_types import PcodeZfpPackage
from sandbox.projects.pcode.zfp.utils import validate_params


class PcodeZfpValidateAlertConfig(binary_task.LastBinaryTaskRelease, sdk2.Task):
    name = "PCODE_ZFP_VALIDATE_ALERT_CONFIG"

    class Parameters(sdk2.Task.Parameters):
        alert_config_name = sdk2.parameters.String('Alert config to validate', default_value="")
        pr_id = sdk2.parameters.String('Pull request id', default_value="")
        alert_package_resource = sdk2.parameters.Resource("Alerts package resource", resource_type=PcodeZfpPackage)
        arcanum_token = sdk2.parameters.YavSecretWithKey(
            "Arcanum token",
            default="sec-01cv39zerwcz9zk1wjxght1zb8#arcanum-token",

        )

        ext_params = binary_task.binary_release_parameters(stable=True)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": PcodeZfpValidateAlertConfig.name, "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def on_execute(self):
        from adv.pcode.zfp.utils.arcanum import publish_pr_comment

        alert_name = self.Parameters.alert_config_name
        arcanum_token = self.Parameters.arcanum_token.data()[self.Parameters.arcanum_token.default_key]
        comment = "Alert config for {} is ok.".format(alert_name)
        is_issue = False
        errors = self.validate()
        if errors:
            comment = '\n'.join(errors)
            is_issue = True
        publish_pr_comment(comment, self.Parameters.pr_id, arcanum_token, is_issue)

    def validate(self):
        alert_name = self.Parameters.alert_config_name
        package = sdk2.ResourceData(self.Parameters.alert_package_resource)
        package_path = str(package.path)
        logging.info(os.getcwd())
        logging.info(package_path)
        with tarfile.open(package_path) as tar:
            tar.extractall()
        config_path = os.path.join(os.getcwd(), "alert_configs/{}/params.yaml".format(alert_name))
        logging.info(config_path)
        if not os.path.exists(config_path):
            raise TaskFailure("{} is not exists".format(config_path))
        with open(config_path) as params_file:
            params = yaml.safe_load(params_file.read())
        return validate_params(params, alert_name)
