# -*- coding: utf-8 -*-
from sandbox import sdk2
import logging
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.projects.resource_types as resource_types
import time
from sandbox.projects.peoplesearch.BuildPPLImagesOneShard import BuildPPLImagesOneShard
import sandbox.common.types.task as ctt
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.search.database.iss_shards import _get_iss_shards_tool

peoplesearch_releasers = ['nkmakarov', 'iv-ivan']


class PPLImagesShardMap(sdk2.Resource):
    releasable = True
    releasers = peoplesearch_releasers
    timestamp = sdk2.Attributes.Integer("Creation timestamp")
    auto_backup = True


class BuildPPLImagesShards(sdk2.Task):
    class Requirements(sdk2.Requirements):
        ram = 100
        disk_space = 100

    class Parameters(sdk2.Task.Parameters):
        index_prefix = sdk2.parameters.String('YT path to index', required=True)
        index_state = sdk2.parameters.String('Index state', required=True)
        shards_num = sdk2.parameters.Integer('Shards total number', required=True)
        timestamp = sdk2.parameters.Integer('Timestamp', required=False, default=int(time.time()))
        yt_server = sdk2.parameters.String('YT server short name', default="banach")
        yt_token_secret_name = sdk2.parameters.String('YT token vault secret', required=True)
        shardwriter = sdk2.parameters.Resource("Shardwriter resource", resource_type=resource_types.IMAGES_MR_INDEX_SHARDWRITER, required=True)
        images_shard = sdk2.parameters.Resource("Images shard resource", resource_type=resource_types.IMAGES_SEARCH_DATABASE, required=True)

    def on_execute(self):
        with self.memoize_stage.stage1(commit_on_entrance=False):
            subtask_params = {
                "index_prefix": self.Parameters.index_prefix,
                "index_state": self.Parameters.index_state,
                "yt_server": self.Parameters.yt_server,
                "timestamp": self.Parameters.timestamp,
                "yt_token_secret_name": self.Parameters.yt_token_secret_name,
                "shardwriter": self.Parameters.shardwriter.id,
                "images_shard": self.Parameters.images_shard.id
            }
            all_subtasks = []
            for i in xrange(self.Parameters.shards_num):
                subtask_params["shard"] = i
                sub_task = BuildPPLImagesOneShard(
                    self,
                    description="Child of task {}".format(self.id),
                    create_sub_task=True,
                    **subtask_params
                )
                requirements = {
                    'disk_space': 105 << 30,  # in bytes, 250 Gb
                    'ram': 10 << 30,  # in bytes, 10 Gb
                }
                sdk2.Task.server.task[sub_task.id].update({'requirements': requirements})
                sub_task.enqueue()
                all_subtasks.append(sub_task.id)
            raise sdk2.WaitTask(all_subtasks, list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK), wait_all=True)

        with self.memoize_stage.stage2(commit_on_entrance=False):
            subtasks = self.find()
            if any(subtask.status != ctt.Status.SUCCESS for subtask in subtasks):
                for subtask in subtasks:
                    if subtask.status != ctt.Status.SUCCESS:
                        raise TaskFailure("Subtask {} finished with status {}".format(subtask.id, subtask.status))
                    else:
                        logging.info("Success!")
            resource = sdk2.ResourceData(PPLImagesShardMap(self, "Shardmap PPL-images", "shardmap.map", timestamp=self.Parameters.timestamp))
            shards_info = ""
            for i in xrange(self.Parameters.shards_num):
                shards_info += "peopleimages-%03d-0000000000 peopleimages-%03d-%s-1 ImgPplTier0\n" % (i, i, self.Parameters.timestamp)
            resource.path.write_bytes(shards_info)
            resource.ready()

            # register shardmap
            iss_shards = sdk2.ResourceData(sdk2.Resource.find(id=_get_iss_shards_tool()).first())
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("iss_shards")) as pl:
                status = sp.Popen(
                    "{} register_shardmap {} {}".format(
                        iss_shards.path, "peopleimages#" + str(self.Parameters.timestamp), resource.path
                    ),
                    shell=True, stdout=pl.stdout, stderr=sp.STDOUT
                ).wait()
                if status != 0:
                    raise Exception("Can't register shardmap")
