# coding: utf-8

import os
import shutil
import textwrap

import sandbox.common.types.misc as ctm
from sandbox.common.types.client import Tag

import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.paths as sdk_paths
from sandbox.sandboxsdk.process import run_process

from sandbox.projects import resource_types
from sandbox.projects.common.commands import get_make_name
from sandbox.projects.common.build.BuildForAllTask import DoNotRemoveResourcesParameter


class CheckoutPortoUrl(sdk_parameters.SandboxGitUrlParameter):
    name = 'checkout_porto_from_url'
    default_value = 'git://git.yandex.ru/search-admin/porto'


class GitTag(sdk_parameters.SandboxStringParameter):
    name = 'git_tag'
    description = 'Git tag or branch name'
    default_value = 'master'
    required = True


class SuspendBeforeMake(sdk_parameters.SandboxBoolParameter):
    name = 'suspend_before_make'
    description = 'Suspend task execution before make'
    default_value = False
    required = True


class BuildPorto(sdk_task.SandboxTask):
    type = 'BUILD_PORTO'
    client_tags = (Tag.GENERIC | Tag.SKYNET) & Tag.LINUX_PRECISE
    input_parameters = [
        CheckoutPortoUrl,
        GitTag,
        DoNotRemoveResourcesParameter,
        SuspendBeforeMake,
    ]
    execution_space = 4 * 1024
    dns = ctm.DnsType.DNS64

    @property
    def privileged(self):
        return True

    def checkout(self):
        url = self.ctx[CheckoutPortoUrl.name]
        tag = self.ctx[GitTag.name]

        path = self.path('porto')
        sdk_paths.make_folder(path, delete_content=True)

        run_process(['git', 'clone', url, path], log_prefix='git_clone').communicate()
        run_process(['git', 'checkout', tag], log_prefix='git_checkout', work_dir=path).communicate()

        rev = run_process(['git', 'symbolic-ref', '-q', 'HEAD'],
                          outs_to_pipe=True,
                          check=False,
                          work_dir=path
                          ).communicate()[0].strip()
        if not rev:
            rev = run_process(['git', 'describe', '--tags', '--exact-match'],
                              outs_to_pipe=True,
                              check=False,
                              work_dir=path
                              ).communicate()[0].strip()
            rev = '<tag>' + rev if rev else tag
        else:
            rev = rev.split('/')[-1]

        with open(self.path('porto.patch'), 'w') as f:
            f.write(
"""--- a/src/CMakeLists.txt
+++ b/src/CMakeLists.txt
@@ -1,5 +1,6 @@
 project(porto)

+set(CURSES_NEED_NCURSES TRUE)
 find_package(Curses REQUIRED)
 include_directories(${CURSES_INCLUDE_DIR})

@@ -52,7 +53,7 @@ target_link_libraries(portod version por

 add_executable(portoctl portoctl.cpp cli.cpp portotop.cpp)
 target_link_libraries(portoctl version porto util
-			       rt ${PB} ${CURSES_LIBRARIES})
+			       pthread rt ${PB} ${CURSES_LIBRARIES})

 add_executable(portoinit portoinit.c)
 target_link_libraries(portoinit version)
"""
            )

        run_process(['patch', '-p1'], stdin=open(self.path('porto.patch')), log_prefix='patch_porto', work_dir=path)

        return rev

    def build(self):
        srcdir = self.path('porto')
        builddir = self.path('build')
        sdk_paths.make_folder(builddir, delete_content=True)

        run_process(['apt-get', 'update'], log_prefix='apt', work_dir=builddir)
        run_process(['apt-get', '-y', 'install',
                     'cmake', 'libncurses5-dev',
                     'dh-autoreconf', 'pkg-config',
                     'bison', 'autoconf', 'libtool', 'flex',
                     'libprotobuf-dev', 'protobuf-compiler',
                     'g++-4.7',
                     ],
                    log_prefix='apt', work_dir=builddir)
        run_process(['cmake',
                     '-DCMAKE_BUILD_TYPE=Release',
                     '-DUSE_SYSTEM_LIBNL=OFF',
                     '-DCMAKE_VERBOSE_MAKEFILE=yes',
                     srcdir],
                    log_prefix='cmake',
                    work_dir=builddir)

        if self.ctx[SuspendBeforeMake.name]:
            self.suspend()

        run_process(get_make_name(True), log_prefix='make', work_dir=builddir)

    def install(self):
        debdir = self.path('porto/debian')
        builddir = self.path('build')
        installdir = self.path('install')
        sdk_paths.make_folder(installdir, delete_content=True)

        run_process(get_make_name(False) + ['install'],
                    log_prefix='install',
                    work_dir=builddir,
                    environment={'DESTDIR': installdir})

        scriptsdir = self.path(os.path.join('install', 'scripts'))
        sdk_paths.make_folder(scriptsdir, delete_content=True)

        etcdir = self.path(os.path.join('install', 'etc'))
        sdk_paths.make_folder(etcdir, delete_content=True)
        shutil.copy(os.path.join(debdir, 'yandex-porto.cron.d'), os.path.join(etcdir, 'cron'))
        shutil.copy(os.path.join(debdir, 'yandex-porto.logrotate'), os.path.join(etcdir, 'logrotate'))
        shutil.copy(os.path.join(debdir, 'yandex-porto.bash-completion'), os.path.join(etcdir, 'bash-completion'))

        with open(os.path.join(scriptsdir, 'start'), 'wb') as start:
            start.write(textwrap.dedent(
                """
                #!/bin/sh
                [ ! -e /run/portoloop.pid ] && rm -rf /run/porto/pkvs /run/porto/kvs /tmp/portod.pidmap || /bin/true
                chown root:porto usr/sbin/portoctl
                exec usr/sbin/portod
                """
            ))

        with open(os.path.join(scriptsdir, 'check'), 'wb') as check:
            check.write(textwrap.dedent(
                """
                #!/bin/sh
                set -e
                portoctl list &>/dev/null
                """
            ))

        with open(os.path.join(scriptsdir, 'stop'), 'wb') as stop:
            stop.write(textwrap.dedent(
                """
                #!/bin/sh
                [ -f /run/portoloop.pid ] && kill -SIGINT $(cat /run/portoloop.pid)
                """
            ))
        with open(os.path.join(scriptsdir, 'install'), 'wb') as install:
            install.write(textwrap.dedent(
                """
                #!/bin/sh
                MODE=$1

                case $MODE in
                    upgrade)
                        [ -f /run/portoloop.pid ] && kill -HUP $(cat /run/portoloop.pid)
                        ;;
                    install)
                        groupadd -f porto
                        usermod -a -G porto loadbase || /bin/true
                esac
                install -m644 etc/cron /etc/cron.d/yandex-porto
                install -m644 etc/logrotate /etc/logrotate.d/yandex-porto
                install -m644 etc/bash-completion /etc/bash_completion.d/yandex-porto
                [ -f /usr/sbin/dpkg-divert ] && /usr/sbin/dpkg-divert --rename --local --add /usr/sbin/portoctl || /bin/true
                chown root:porto usr/sbin/portoctl
                ln -Tfs $PWD/usr/sbin/portoctl /usr/sbin/portoctl
                """
            ))
        with open(os.path.join(scriptsdir, 'uninstall'), 'wb') as uninstall:
            uninstall.write(textwrap.dedent(
                """
                #!/bin/sh
                MODE=$1

                case $MODE in
                    preupgrade)
                        # currently do nothing
                        ;;
                    uninstall)
                        [ -f /usr/sbin/dpkg-divert ] && /usr/sbin/dpkg-divert --rename --local --remove /usr/sbin/portoctl || /bin/true
                        ;;
                esac
                """
            ))
        with open(os.path.join(installdir, 'porto.scsd'), 'wb') as scsd:
            scsd.write(textwrap.dedent(
                """
                conf: []
                require: []
                exec:
                    - /bin/sh scripts/start
                check: /bin/sh scripts/check
                stop: /bin/sh scripts/stop
                porto: no
                restart_on_upgrade: false
                install_script: /bin/sh scripts/install ${MODE}
                uninstall_script: /bin/sh scripts/uninstall ${MODE}
                user: root
                cgroup: null
                limits: {}
                env: {}
                """
            ))

    def archive(self):
        path = self.path('install')
        run_process(['tar', '-C', path, '-czvf', 'porto.tgz', '.'], log_prefix='archive')

    def on_execute(self):
        rev = self.checkout()
        self.build()
        self.install()
        self.archive()

        self.create_resource(
            'porto (%s %s)' % (self.ctx[GitTag.name], rev),
            self.path('porto.tgz'),
            resource_types.SKYCORE_SERVICE,
            arch=self.arch,
        )


__Task__ = BuildPorto
