import os
import re
import logging
import sandbox.projects.common.utils as sandbox_utils

from sandbox import common
from sandbox import sdk2
from sandbox.sdk2.helpers.process import subprocess

from sandbox import sandboxsdk
from sandbox.common.types.client import Tag
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.projects.porto.BuildPortoLayer import (
    BuildPortoLayer, ParentLayerType, ParentLayer,
    LayerType, LayerName, Compress, ScriptPack,
    BootstrapUrl, Bootstrap2Url, ScriptUrl,
    Script2Url, ScriptEnv, VaultEnv, StartOS,
    KeepResolvConf, MergeLayers, SpaceLimit,
    MemoryLimit, DebugBuild, OutputResourceID,
    OutputResourceAttr, UseNat
)


class UseInheritedNet(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'inherited_network'
    description = 'Use inherited network'
    default_value = True
    required = True


class BuildPortoLayerTmp(BuildPortoLayer):
    type = 'BUILD_PORTO_LAYER_TMP'
    client_tags = Tag.PORTOD
    privileged = True
    input_parameters = [
        ParentLayerType,
        ParentLayer,
        LayerType,
        LayerName,
        Compress,
        ScriptPack,
        BootstrapUrl,
        Bootstrap2Url,
        ScriptUrl,
        Script2Url,
        ScriptEnv,
        VaultEnv,
        StartOS,
        KeepResolvConf,
        MergeLayers,
        SpaceLimit,
        MemoryLimit,
        DebugBuild,
        OutputResourceID,
        OutputResourceAttr,
        UseNat,
        UseInheritedNet,
    ]

    def on_execute(self):
        layer_type = self.ctx[LayerType.name]
        layer_name = self.ctx[LayerName.name]
        output_file = layer_name + '.' + self.ctx[Compress.name]
        parent_layer = self.ctx[ParentLayer.name]
        script_pack_url = self.ctx[ScriptPack.name]
        bootstrap_url = self.ctx[BootstrapUrl.name]
        bootstrap2_url = self.ctx[Bootstrap2Url.name]
        build_image = layer_type.startswith('PORTO_IMAGE')
        build_qcow = layer_type.startswith('QEMU_IMAGE')
        build_squash = layer_type.startswith('PORTO_SQUASH')
        build_lxc = layer_type == 'LXC_CONTAINER'
        merge_layers = self.ctx[MergeLayers.name] or build_lxc
        output_resource_id = self.ctx.get(OutputResourceID.name)
        debug_build = self.ctx.get(DebugBuild.name)

        scripts = []
        env = {}
        has_bootstrap = bool(bootstrap_url)

        cmd = ['portoctl', '--timeout', '0', 'build']

        if script_pack_url:
            script_pack = self.abs_path('script_pack')
            self.remote_copy(script_pack_url, script_pack)
            for phase, phase_opt in [('/bootstrap', '-B'), ('/pre-os', '-b'), ('', '-S'), ('/post-os', '-s')]:
                phase_path = script_pack + phase
                if os.path.exists(phase_path):
                    names = os.listdir(phase_path)
                    names.sort()
                    for name in names:
                        script_path = phase_path + '/' + name
                        if name.endswith('.sh') and os.path.isfile(script_path):
                            scripts += [(script_pack_url, script_path)]
                            cmd += [phase_opt, script_path]
                            if phase_opt == '-B':
                                has_bootstrap = True

        if bootstrap_url:
            bootstrap_path = self.abs_path('bootstrap.sh')
            self.remote_copy(bootstrap_url, bootstrap_path)
            content = open(bootstrap_path, 'U').read()
            open(bootstrap_path, 'w').write(content)
            cmd += ['-B', bootstrap_path]

        if bootstrap2_url:
            bootstrap2_path = self.abs_path('bootstrap2.sh')
            self.remote_copy(bootstrap2_url, bootstrap2_path)
            content = open(bootstrap2_path, 'U').read()
            open(bootstrap2_path, 'w').write(content)
            cmd += ['-b', bootstrap2_path]

        vault_env = {}
        if self.ctx[VaultEnv.name]:
            vault_env.update(self.ctx[VaultEnv.name])

        vault_script = ''
        for vault_var, vault_key in vault_env.items():
            logging.info('Acquiring vault key {}'.format(vault_key))
            if ':' in vault_key:
                vault_owner, vault_key = vault_key.split(':', 1)
            else:
                vault_owner, vault_key = vault_key, None
            vault_data = self.get_vault_data(vault_owner, vault_key)
            if len(vault_data) == 0:
                raise Exception('Vault key acquiring failed')
            vault_escaped = "'" + str(vault_data).replace("'", "'\\''") + "'"
            vault_script += "{}={}\n".format(vault_var, vault_escaped)

            vault_filter = common.log.VaultFilter.filter_from_logger(logging.getLogger())
            if vault_filter:
                vault_filter.add_record(vault_var + '_ESCAPED', vault_escaped)

        # Sandbox API token with task privileges. Authenticates as task author. See SANDBOX-7655.
        # Layer for vault script breaks debugging
        if not debug_build:
            vault_script += "{}={}\n".format('sandbox_task_token', common.rest.Client._external_auth.task_token)

        for script_url in filter(None, re.split('[\r\n]+', self.ctx[ScriptUrl.name])):
            script_path = self.abs_path('script{}.sh'.format(len(scripts)))
            self.remote_copy(script_url, script_path)
            script_text = file(script_path, 'U').read()
            open(script_path, 'w').write(script_text)
            scripts += [(script_url, script_path)]
            cmd += ['-S', script_path]

        for script_url in filter(None, re.split('[\r\n]+', self.ctx[Script2Url.name])):
            script_path = self.abs_path('script{}.sh'.format(len(scripts)))
            self.remote_copy(script_url, script_path)
            script_text = file(script_path, 'U').read()
            open(script_path, 'w').write(script_text)
            scripts += [(script_url, script_path)]
            cmd += ['-s', script_path]

        if has_bootstrap or build_lxc:
            logging.info('Get latest porto bootstrap layer')
            bootstrap_stable = channel.sandbox.list_releases(
                resource_type='PORTO_LAYER_BOOTSTRAP',
                arch='linux',
                release_status='stable',
                limit=1)[0].resources[0]
            bootstrap_tarball = self.sync_resource(bootstrap_stable.id)

        if has_bootstrap:
            cmd += ['-l', bootstrap_tarball]

        parent_resources = self._parent_resources()
        for resource in parent_resources:
            layer_path_tmp = self.sync_resource(resource.id)
            cmd += ['-L', layer_path_tmp]

        if build_squash:
            output_file = layer_name + '.squash'
            cmd += ['-Q', output_file]
        elif build_image:
            cmd += ['-O', layer_name + '.img']
        elif build_qcow:
            output_file = layer_name + '.img'
            cmd += ['-O', layer_name + '.raw']
        else:
            cmd += ['-o', output_file]

        if merge_layers:
            cmd += ['-M']

        # keep volume and container for manual debugging
        if debug_build:
            cmd += ['-k']

        if self.ctx[ScriptEnv.name]:
            env.update(self.ctx[ScriptEnv.name])

        # write vault script only after all other preparations
        vault_path = ''
        if vault_script:
            vault_path = os.path.join(self.abs_path('secrets'), 'secret_env')
            os.mkdir(os.path.dirname(vault_path))
            vaultfd = open(vault_path, 'w')
            # do not propagate secret env into nested bash invocations
            vaultfd.write('set +x\n' + vault_script + 'unset BASH_ENV\nunset ENV\nset -x\n')
            vaultfd.close()
            cmd += ['-L', os.path.dirname(vault_path)]
            env.update({'BASH_ENV': os.path.basename(vault_path), 'ENV': os.path.basename(vault_path)})
            vault_script = ''

        env['sandbox_task_id'] = self.id

        if env:
            cmd += ['env=' + ';'.join(['{}={}'.format(k, v).replace(';', '\;') for k, v in env.iteritems()])]

        cmd += ['bind_dns=false']

        if self.ctx.get(KeepResolvConf.name):
            cmd += ['resolv_conf=keep']
        # TODO: net
        cmd += ['net=inherited']

        cmd += ['aging_time=3600',
                'enable_porto=false',
                'hostname=sandbox-{}'.format(self.id),
                'virt_mode=' + ('os' if self.ctx[StartOS.name] else 'app'),
                'private=sandbox-{}'.format(self.id),
                'space_limit={}M'.format(self.ctx[SpaceLimit.name]),
                'memory_limit={}M'.format(self.ctx[MemoryLimit.name])]

        with sdk2.helpers.ProcessLog(logger="build") as pl:
            try:
                subprocess.check_call(cmd, stdout=pl.stdout, stderr=subprocess.STDOUT)
            except subprocess.CalledProcessError as e:
                try:
                    self.set_info(''.join(open(get_logs_folder() + '/build.out.log').readlines()[-50:]))
                except:
                    pass
                raise e
            finally:
                if vault_path:
                    os.unlink(vault_path)

        if build_image:
            run_process(['tar', 'cSaf', output_file, layer_name + '.img'], log_prefix='tar-img')

        if build_qcow:
            volume_path = self.abs_path('image')
            disk_img = self.abs_path('disk.raw')
            image_size = os.path.getsize(layer_name + '.raw')
            gptbin = ""
            gptbin_search = ['/usr/lib/syslinux/gptmbr.bin', '/usr/lib/EXTLINUX/gptmbr.bin', '/usr/lib/SYSLINUX/gptmbr.bin']
            for f in gptbin_search:
                if os.path.exists(f):
                    gptbin = f
            if not gptbin:
                raise Exception('Can not fild bootloader stage1 binnary: gptmbr.bin')

            di = open(disk_img, 'a')
            di.truncate(image_size + (2 << 20))
            di.close()
            # Create GPT partitions
            # NOTE: Do not set partition label name because of https://st.yandex-team.ru/QEMUKVM-379
            run_process(['sgdisk', '--clear', '-g', '--new', '1::-0', '--typecode=1:8300', disk_img], log_prefix='qemu_image')
            run_process(['sgdisk', '--attributes=1:set:2', disk_img], log_prefix='qemu_image')
            # Install bootloader stage1
            run_process(['dd', 'if=' + gptbin, 'of=' + disk_img, 'conv=notrunc', 'bs=440', 'count=1'], log_prefix='qemu_image')
            run_process(['dd', 'if=' + layer_name + '.raw', 'of=' + disk_img, 'conv=notrunc', 'bs=1048576', 'seek=1'], log_prefix='qemu_image')
            p = run_process(['sudo', 'losetup', '--find', '--show', disk_img], stdout=subprocess.PIPE, log_prefix='qemu_image')
            loopdev = p.stdout.readline().strip()
            run_process(['sudo', 'partprobe', loopdev],  log_prefix='qemu_image')
            # Double check that partprobe have found partition
            assert os.path.exists(loopdev + 'p1')
            # Create ext4 journal
            run_process(['sudo', 'tune2fs', '-O', 'has_journal', loopdev + 'p1'], log_prefix='qemu_image')
            # Install bootloader stage2
            os.mkdir(volume_path)
            run_process(['sudo', 'mount', loopdev + 'p1', volume_path],  log_prefix='qemu_image')
            run_process(['sudo', 'extlinux', '--install', volume_path], log_prefix='qemu_image')
            run_process(['sudo', 'umount', loopdev + 'p1'], log_prefix='qemu_image')
            os.rmdir(volume_path)
            run_process(['sudo', 'tune2fs', '-L', 'rootfs', loopdev + 'p1'], log_prefix='qemu_image')
            run_process(['sudo', 'e2fsck', '-f', '-p', '-E', 'discard', loopdev + 'p1'], log_prefix='qemu_image')
            run_process(['sudo', 'losetup', '-d', loopdev], log_prefix='qemu_image')
            run_process(['qemu-img', 'convert'] +
                        (['-c'] if Compress.name != 'tar' else []) +
                        ['-f', 'raw', '-O', 'qcow2', disk_img, output_file], log_prefix='qemu_image')
            os.unlink(layer_name + '.raw')
            os.unlink(disk_img)

        if build_lxc:
            run_process(['portoctl', '--timeout', '0', 'exec',
                         'sandbox-{}-tar-lxc'.format(self.id),
                         'layers={}'.format(bootstrap_tarball),
                         'user=root',
                         'bind={} {}'.format(self.abs_path(output_file), output_file),
                         'command=sh -c "set -ex; mkdir rootfs; tar --numeric-owner -xaf {0} -C rootfs; tar --numeric-owner -caf {0} rootfs"'.format(output_file)],
                        log_prefix='tar-lxc')

        resource_attributes = {}
        if self.ctx[OutputResourceAttr.name]:
            resource_attributes.update(self.ctx[OutputResourceAttr.name])

        if not merge_layers and not build_image and parent_layer:
            resource_attributes['parent_layer'] = parent_layer
            resource_attributes['dependencies'] = " ".join([str(r.id) for r in parent_resources])

        resource_attributes['name'] = layer_name
        resource_attributes['stack'] = layer_name
        resource_attributes["compression"] = self.ctx[Compress.name]
        for resource in parent_resources:
            resource_attributes['stack'] += " " + resource.attributes.get('name', 'unknown')

        packed_size = os.path.getsize(output_file)
        try:
            unpacked_size = int(subprocess.check_output(["getfattr", "-n", "user.porto.unpacked_size", "--only-values", output_file]))
        except:
            unpacked_size = 0

        logging.info('Packed size {}M, unpacked size {}M'.format(packed_size >> 20, unpacked_size >> 20))
        self.set_info('Packed size {}M, unpacked size {}M'.format(packed_size >> 20, unpacked_size >> 20))

        if unpacked_size:
            resource_attributes["unpacked_size"] = str(unpacked_size)

        if output_resource_id is not None:
            sandbox_utils.set_resource_attributes(output_resource_id, resource_attributes)
            self.save_parent_task_resource(output_file, output_resource_id)
        else:
            self.create_resource(self.descr, output_file, layer_type,
                                 arch='linux', attributes=resource_attributes)

        if bootstrap_url:
            self.create_resource(bootstrap_url, bootstrap_path, 'PLAIN_TEXT', arch='linux')

        if bootstrap2_url:
            self.create_resource(bootstrap2_url, bootstrap2_path, 'PLAIN_TEXT', arch='linux')

        for script_url, script_path in scripts:
            self.create_resource(script_url, script_path, 'PLAIN_TEXT', arch='linux')
