# coding: utf-8
import jinja2
import os
import re

from sandbox import sdk2
from sandbox import common

from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils2
from sandbox.projects.prs_ops.resources import PRS_OPS_DIFF as output_resource

_first = 0
_second = 1
_splitable = frozenset(["rearr", "pron", "relev"])
_same_key = "same_key"
_diff_key = "diff_key"
_banned_param = frozenset(["reqid", "ruid", "?_prs_ops.hostname", "yandexuid", "req_path",
                           "app_host",  # https://paste.yandex-team.ru/274162
                           ])
_banned_item = [("rearr", re.compile("ARRAY.*"))]


class PrsOpsCompare(sdk2.Task):
    """
    Compare two outputs and make diff
    """

    class Requirements(sdk2.Task.Requirements):
        ram = 40 * 1024
        disk_space = 40 * 1024

    class Parameters(sdk2.Task.Parameters):
        prs_ops_req1 = sdk2.parameters.Resource(
            "prs_ops patched requests 1",
            # resourse_type=input_resource,  # FIXME: invalid argument (SANDBOX-6404)
            required=True,
        )
        prs_ops_req2 = sdk2.parameters.Resource(
            "prs_ops patched requests 2",
            # resourse_type=input_resource,  # FIXME: invalid argument (SANDBOX-6404)
            required=True,
        )
        # with sdk2.parameters.Output:
        #     has_diff = sdk2.parameters.Bool("has_diff", default=False, required=True)
        #     can_compare = sdk2.parameters.Bool("can_compare", default=True, required=True)
        prs_ops_fail_on_diff = sdk2.parameters.Bool("fail on diff", default=False)

    @staticmethod
    def _get_diff(dif_param):
        diff_res = {}
        for expr in dif_param:
            params = expr.split("=", 1)
            value_list = diff_res.setdefault(params[_first], [])
            if params[_first] not in _banned_param:
                if params[_first] in _splitable:
                    value_list.extend(params[_second].split(";"))
                else:
                    value_list.append(params[_second])
        for key in diff_res:
            for item in _banned_item:
                if key == item[0]:
                    diff_res[key] = filter(lambda x: not (item[1].search(x)), diff_res[key])
        return diff_res

    @staticmethod
    def _make_set_params(diff_res):
        set_res = set()
        for key in diff_res:
            for val in diff_res[key]:
                set_res.add((key, val))
        return set_res

    @staticmethod
    def _param_for_html(diff_res, diff_key_params, diff_items):
        result = {_diff_key: {}, _same_key: {}}
        for item in diff_res:
            if item in zip(*diff_items)[_first]:
                result[_same_key][item] = []
                result[_diff_key][item] = []
                for val in diff_res[item]:
                    if (item, val) in diff_items:
                        if item in diff_key_params:
                            result[_diff_key][item].append(val)
                        else:
                            result[_same_key][item].append(val)
        return result

    @staticmethod
    def _create_html(first, second, req1, req2):
        html_path = os.path.join(
            # os.path.dirname(sandbox.projects.prs_ops.PrsOpsCompare.__file__),   UNTIL SM1 WANT THIS TASK
            "template_diff_prs_ops.html"
        )
        template = fu.read_file(html_path)
        out_html = jinja2.Template(template).render(
            first=first,
            second=second,
            num=len(first),
            xrange=xrange,
            req1=req1,
            req2=req2,
        ).encode("utf-8")
        return out_html

    def on_execute(self):

        self.Context.has_diff = False
        self.Context.can_compare = True

        req1 = fu.read_lines(sdk2.ResourceData(self.Parameters.prs_ops_req1).path)
        req2 = fu.read_lines(sdk2.ResourceData(self.Parameters.prs_ops_req2).path)

        req_number = len(req1)

        if self.Parameters.prs_ops_req1.md5 == self.Parameters.prs_ops_req2.md5:
            self.set_info("Files has same md5")
            return

        task1 = sdk2.Task[self.Parameters.prs_ops_req1.task_id]
        task2 = sdk2.Task[self.Parameters.prs_ops_req2.task_id]

        if str(task1.type) != "RUN_PRS_OPS" or str(task2.type) != "RUN_PRS_OPS":
            self.Context.can_compare = False
            self.Context.has_diff = True
            self.set_info("Resources got from incompatible task, first {}, second {}".format(task1.type, task2.type))
            return

        if (task1.Parameters.prs_ops_queries_resource != task2.Parameters.prs_ops_queries_resource) or \
                (task2.Parameters.prs_ops_ratings_resource != task2.Parameters.prs_ops_ratings_resource):
            self.Context.can_compare = False
            self.Context.has_diff = True
            self.set_info("requests was generated from different sources")
            return

        if req_number != len(req2):
            self.set_info("Files has different number of lines! First {},"
                          " second {} Something go wrong (maybe)".format(req_number, len(req2)))
            if self.Parameters.prs_ops_fail_on_diff:
                raise common.errors.TaskFailure("Files has different number of lines")
            return

        first = []
        second = []

        has_diff = False

        txt = re.compile("(?<=text=).*?&")
        req1.sort(key=lambda x: txt.search(x).group(0))
        req2.sort(key=lambda x: txt.search(x).group(0))

        for i in xrange(req_number):

            diff_res_1 = self._get_diff(req1[i].split("&"))
            diff_res_2 = self._get_diff(req2[i].split("&"))

            set_res_1 = self._make_set_params(diff_res_1)
            set_res_2 = self._make_set_params(diff_res_2)

            diff_key_params = (set(diff_res_1.keys()) ^ set(diff_res_2.keys()))
            diff_items = set_res_1 ^ set_res_2

            if diff_key_params or diff_items:
                has_diff = True
                first.append((i, self._param_for_html(diff_res_1, diff_key_params, diff_items)))
                second.append((i, self._param_for_html(diff_res_2, diff_key_params, diff_items)))

        out_html = self._create_html(first, second, req1, req2)
        out_resource = output_resource(self, "diff", "diff.html")
        out_resource_data = sdk2.ResourceData(out_resource)

        fu.write_file(out_resource_data.path, out_html)
        out_resource_data.ready()

        if has_diff:
            self.Context.has_diff = True
            diff_html = '<a href="{}" target="_blank">diff</a>'.format(
                utils2.resource_redirect_url(out_resource.id)
            )
            self.set_info('There is a diff! See {}'.format(diff_html), do_escape=False)
            if self.Parameters.prs_ops_fail_on_diff:
                raise common.errors.TaskFailure('We has some diff')
            return

        self.set_info("There is no diff!")
