# coding: utf-8

import os
import re
import logging
from sandbox import sdk2
import traceback

from sandbox.sandboxsdk import svn
# from sandbox.projects.prs_ops import resources
from sandbox.projects.prs_ops import components
from sandbox.common.types.client import Tag
from sandbox.sandboxsdk import process
# from sandbox.projects import resource_types
from sandbox.projects.common.search import bugbanner2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components.configs.prs_ops import PrsOpsCfg
from sandbox.projects.release_machine import rm_notify as rm_notify
from sandbox.projects.common.sdk_compat import task_helper

_ADDRESS_SANITIZER_OUTPUT = re.compile(r"AddressSanitizer Init done")
_MEMORY_SANITIZER_OUTPUT = re.compile(r"MemorySanitizer init done")
_THREAD_SANITIZER_OUTPUT = re.compile(r"^\*\*\*\*\* Running under ThreadSanitizer v2 \(pid \d+\) \*\*\*\*\*")


@rm_notify.notify2()
class RunPrsOps(bugbanner2.BugBannerTask):
    """
    Run prs_ops, get ratings and queries as input and return patched_requests as output resource
    """

    class Requirements(sdk2.Task.Requirements):
        ram = 40 * 1024
        disk_space = 5 * 1024
        # clients in LXC containers don't have host names, only v6 address
        client_tags = Tag.GENERIC & Tag.Group.LINUX & ~Tag.LXC

        environments = [
            PipEnvironment('yandex-yt', version="0.8.29.post0", use_wheel=True),
            PipEnvironment('yandex-yt-yson-bindings-skynet', version="0.3.7.post1", use_wheel=True)
        ]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 8 * 60 * 60
        substitute_ranks = sdk2.parameters.Bool("substitute-ranks", default=True)
        is_queries_full = sdk2.parameters.Bool("is queries-format-full", default=True)
        gdtsf = sdk2.parameters.Bool("enable gather-dynamic-top-sizes-factors", default=False)
        customCGI = sdk2.parameters.String("customCGI")
        selected_slices = sdk2.parameters.String("selected-slices")
        mode = sdk2.parameters.String("Mode to run", default="COMBO", required=True)
        mr_server = sdk2.parameters.String("mr-server", default="hahn.yt.yandex.net")
        save_to = sdk2.parameters.String(
            "result folder on server (dir in //home/prs_ops/)",
            default="111111"
        )
        disable_samohod = sdk2.parameters.Bool("disable samohod on middle", default=True)
        write_mode = sdk2.parameters.String("write mode", default="mr-tsv")
        success = sdk2.parameters.Float("success threshold", default=0.9)
        tm = sdk2.parameters.Bool("add text machine params", default=False)
        plus = sdk2.parameters.Bool("using prs-plus option", default=False)
        judged_urls = sdk2.parameters.Bool("TMHits only for judged urls", default=False)
        converter = sdk2.parameters.Resource(
            "pool_converter executable",
            # resourse_type=resource_types.POOL_CONVERTER_EXECUTABLE,  # FIXME: invalid argument (SANDBOX-6404)
        )
        reader = sdk2.parameters.Resource(
            "proto_pool_reader executable",
            # resourse_type=resource_types.PROTO_POOL_READER_EXECUTABLE,  # FIXME: invalid argument (SANDBOX-6404)
        )
        mapreduce = sdk2.parameters.Resource(
            "mapreduce_yt executable",
            # resourse_type=resource_types.MAPREDUCE_YT_EXECUTABLE,  # FIXME: invalid argument (SANDBOX-6404)
        )

        with sdk2.parameters.String("Sanitizer type", multiline=True) as sanitizer_type:
            sanitizer_type.values['none'] = sanitizer_type.Value('none', default=True)
            sanitizer_type.values['address'] = sanitizer_type.Value('address')
            sanitizer_type.values['memory'] = sanitizer_type.Value('memory')
            sanitizer_type.values['thread'] = sanitizer_type.Value('thread')
            sanitizer_type.values['leak'] = sanitizer_type.Value('leak')
            sanitizer_type.values['undefined'] = sanitizer_type.Value('undefined')

        args = sdk2.parameters.String("additional command line parameters:")

        prs_ops_binary_resource = sdk2.parameters.Resource(
            "prs_ops executable",
            # resourse_type=resources.PRS_OPS_EXECUTABLE,  # FIXME: invalid argument (SANDBOX-6404)
            required=True,
        )
        prs_ops_queries_resource = sdk2.parameters.Resource(
            "prs_ops queries",
            # resourse_type=resources.PRS_OPS_QUERIES,  # FIXME: invalid argument (SANDBOX-6404)
            required=True,
        )
        prs_ops_ratings_resource = sdk2.parameters.Resource(
            "prs_ops ratings",
            # resourse_type=resources.PRS_OPS_RATINGS,  # FIXME: invalid argument (SANDBOX-6404)
            required=True,
        )

    def on_enqueue(self):
        self.Context.save_to = "{}_id={}".format(self.Parameters.save_to, self.id)
        task_helper.ctx_field_set(self, rm_const.COMPONENT_CTX_KEY, PrsOpsCfg.name)

    def on_execute(self):

        self.add_bugbanner(bugbanner2.Banners.PrsOps)

        if "mr" in self.Parameters.write_mode:
            token = sdk2.Vault.data('prs_ops_yt_test')
            from yt.wrapper import YtClient
            prs_ops_env = dict()
            prs_ops_env["YT_PREFIX"] = "/"
            prs_ops_env["YT_TOKEN"] = token
            yt_client = YtClient('hahn', token)
        else:
            yt_client = None
            prs_ops_env = dict()

        prs_ops = components.PrsOpsComponent(
            mode=self.Parameters.mode,
            mr_server=self.Parameters.mr_server,
            save_to="//home/prs_ops/{}".format(self.Context.save_to),
            disable_samohod=self.Parameters.disable_samohod,
            substitute_ranks=self.Parameters.substitute_ranks,
            is_queries_full=self.Parameters.is_queries_full,
            args=self.Parameters.args,
            customCGI=self.Parameters.customCGI,
            selected_slices=self.Parameters.selected_slices,
            binary=self.Parameters.prs_ops_binary_resource,
            queries=self.Parameters.prs_ops_queries_resource,
            ratings=self.Parameters.prs_ops_ratings_resource,
            write_mode=self.Parameters.write_mode,
            success_threshold=self.Parameters.success,
            tm=self.Parameters.tm,
            plus=self.Parameters.plus,
            judged_urls=self.Parameters.judged_urls,
            task=self,
            reader=self.Parameters.reader,
            converter=self.Parameters.converter,
            mapreduce=self.Parameters.mapreduce,
            gdtsf=self.Parameters.gdtsf,
            yt_client=yt_client,
        )

        if self.Parameters.sanitizer_type != "none":
            self._configure_sanitizer_environment(prs_ops, self.Parameters.sanitizer_type)

        if "mr" in self.Parameters.write_mode:
            tmp_env = prs_ops.get_environment()
            tmp_env.update(prs_ops_env)
            prs_ops.set_environment(tmp_env)

        with prs_ops:
            pass

    def _configure_sanitizer_environment(self, component, sanitizer_type):
        env = component.get_environment()
        env.update(get_sanitizer_environment())
        add_tsan_suppressions(env)
        component.set_environment(env)


def get_sanitizer_environment():
    # set options for all sanitizers together
    # it does not matter because only one sanitizer type can be used during build
    env = {
        "ASAN_OPTIONS": "symbolize=1",
        "MSAN_OPTIONS": "symbolize=1",
        "TSAN_OPTIONS": "symbolize=1",
        "LSAN_OPTIONS": "symbolize=1",
        "UBSAN_OPTIONS": "symbolize=1",
    }

    try:
        symbolizer = get_symbolizer()
        env.update({
            "ASAN_SYMBOLIZER_PATH": symbolizer,
            "MSAN_SYMBOLIZER_PATH": symbolizer,
            "TSAN_SYMBOLIZER_PATH": symbolizer,
            "LSAN_SYMBOLIZER_PATH": symbolizer,
            "UBSAN_SYMBOLIZER_PATH": symbolizer,
        })
        external_path = " external_symbolizer_path={}".format(symbolizer)
        env["ASAN_OPTIONS"] += external_path
        env["MSAN_OPTIONS"] += external_path
        env["TSAN_OPTIONS"] += external_path
        env["LSAN_OPTIONS"] += external_path
        env["UBSAN_OPTIONS"] += external_path
    except:
        logging.error("Failed to obtain llvm-symbolizer:\n%s", traceback.format_exc())

    return env


def add_tsan_suppressions(env):
    try:
        path_to_supp = os.path.abspath("tsan_suppressions")
        svn.Arcadia.export("arcadia:/arc/trunk/arcadia/quality/query_pool/prs_ops/tsan_suppressions", path_to_supp)
        env["TSAN_OPTIONS"] += " suppressions={}".format(path_to_supp)
    except Exception:
        logging.error("Failed to add tsan suppressions:\n%s", traceback.format_exc())


def get_symbolizer():
    # this code will download compiler, but it's quite fast (less than a minute)
    dir_name = 'ya_get_symbolizer'
    if not os.path.isdir(dir_name):
        svn.Arcadia.checkout('arcadia:/arc/trunk/arcadia/', dir_name, depth="files")
    symbolizer_retrieve_cmd = [os.path.join(dir_name, 'ya'), 'tool', 'c++', '--print-path']
    p = process.run_process(symbolizer_retrieve_cmd, outs_to_pipe=True)
    output, _ = p.communicate()
    return os.path.join(os.path.dirname(output.strip()), 'llvm-symbolizer')
