# coding: utf-8
import logging
import os

from sandbox import sdk2

from sandbox.projects.prs_ops import components
from sandbox.common.types.client import Tag
from sandbox.projects.common.search import bugbanner2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.paths import make_folder
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components.configs.prs_ops import PrsOpsCfg
from sandbox.projects.release_machine import rm_notify as rm_notify
from sandbox.projects.common.sdk_compat import task_helper

TABLES = {
    "web": [
        "factor_slices", "features", "requests", "patched_requests", "requests.PlatinumTier0",
        "requests.PlatinumTier0.factor",
        "requests.WebTier0", "requests.WebTier0.factor", "requests.WebTier1", "requests.WebTier1.factor"
    ],
    "images": [
        "factor_slices", "features", "requests", "patched_requests", "requests.ImgTier0", "requests.ImgTier0.factor"
    ],
    "video": [
        "factor_slices", "features", "requests", "patched_requests", "requests.VideoTier0", "requests.VideoTier0.factor"
    ]
}


@rm_notify.notify2()
class RunPrsOpsCheckFiles(bugbanner2.BugBannerTask):
    """
    Run prs_ops, get ratings and queries as input and return patched_requests as output resource
    """

    class Requirements(sdk2.Task.Requirements):
        ram = 40 * 1024
        disk_space = 5 * 1024
        # clients in LXC containers don't have host names, only v6 address
        client_tags = Tag.GENERIC & Tag.Group.LINUX & ~Tag.LXC

        environments = [
            PipEnvironment('yandex-yt', version="0.8.29.post0", use_wheel=True),
            PipEnvironment('yandex-yt-yson-bindings-skynet', version="0.3.7.post1", use_wheel=True)
        ]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 5 * 60 * 60
        converter = sdk2.parameters.Resource(
            "pool_converter executable",
            # resourse_type=resource_types.POOL_CONVERTER_EXECUTABLE,  # FIXME: invalid argument (SANDBOX-6404)
        )
        reader = sdk2.parameters.Resource(
            "proto_pool_reader executable",
            # resourse_type=resource_types.PROTO_POOL_READER_EXECUTABLE,  # FIXME: invalid argument (SANDBOX-6404)
        )
        mapreduce = sdk2.parameters.Resource(
            "mapreduce_yt executable",
            # resourse_type=resource_types.MAPREDUCE_YT_EXECUTABLE,  # FIXME: invalid argument (SANDBOX-6404)
        )
        prs_ops_binary_resource = sdk2.parameters.Resource(
            "prs_ops executable",
            # resourse_type=resources.PRS_OPS_EXECUTABLE,  # FIXME: invalid argument (SANDBOX-6404)
            required=True,
        )
        prs_ops_queries_resource = sdk2.parameters.Resource(
            "prs_ops queries",
            # resourse_type=resources.PRS_OPS_QUERIES,  # FIXME: invalid argument (SANDBOX-6404)
            required=True,
        )
        prs_ops_ratings_resource = sdk2.parameters.Resource(
            "prs_ops ratings",
            # resourse_type=resources.PRS_OPS_RATINGS,  # FIXME: invalid argument (SANDBOX-6404)
            required=True,
        )

    def on_enqueue(self):
        task_helper.ctx_field_set(self, rm_const.COMPONENT_CTX_KEY, PrsOpsCfg.name)

    def on_execute(self):

        self.add_bugbanner(bugbanner2.Banners.PrsOps)

        token = sdk2.Vault.data('prs_ops_yt_test')
        from yt.wrapper import YtClient
        prs_ops_env = dict()
        prs_ops_env["YT_PREFIX"] = "/"
        prs_ops_env["YT_TOKEN"] = token
        yt_client = YtClient('hahn', token)

        for search_type in ["web", "images", "video"]:

            origin_workdir = os.getcwd()
            make_folder(os.path.join(origin_workdir, search_type))
            os.chdir(os.path.join(origin_workdir, search_type))

            prs_ops = components.PrsOpsComponent(
                name="prs_ops_tsv_{}".format(search_type),
                mode="COMBO",
                is_queries_full=True,
                args="--search-type {} --app-host-mode --save-sub-requests --no-restart".format(search_type),
                binary=self.Parameters.prs_ops_binary_resource,
                queries=self.Parameters.prs_ops_queries_resource,
                ratings=self.Parameters.prs_ops_ratings_resource,
                write_mode="tsv",
                task=self,
                need_check=False,
            )

            tmp_env = prs_ops.get_environment()
            tmp_env.update(prs_ops_env)
            prs_ops.set_environment(tmp_env)

            with prs_ops:
                pass

            for root, dirnames, filenames in os.walk(origin_workdir):
                for name in filenames:
                    logging.debug("files : {}\n".format(os.path.join(root, name)))

            for file_to_check in TABLES[search_type]:
                try:
                    text = fu.read_file("{}.tsv".format(os.path.join(origin_workdir, search_type, file_to_check)))
                except Exception:
                    eh.check_failed("{} doesn't exist for {} mode".format(file_to_check, search_type))
                if not text:
                    eh.check_failed("{} is empty for {} mode".format(file_to_check, search_type))

            os.chdir(origin_workdir)

        for search_type in ["web", "images", "video"]:
            table_path = "//tmp/prs_ops/{}_{}".format(self.id, search_type)
            prs_ops = components.PrsOpsComponent(
                name="prs_ops_mr_tsv_{}".format(search_type),
                mr_server="hahn.yt.yandex.net",
                save_to=table_path,
                mode="COMBO",
                is_queries_full=True,
                args="--search-type {} --app-host-mode --save-sub-requests --no-restart".format(search_type),
                binary=self.Parameters.prs_ops_binary_resource,
                queries=self.Parameters.prs_ops_queries_resource,
                ratings=self.Parameters.prs_ops_ratings_resource,
                write_mode="mr-tsv",
                task=self,
                reader=self.Parameters.reader,
                converter=self.Parameters.converter,
                mapreduce=self.Parameters.mapreduce,
                yt_client=yt_client,
                need_check=False,
            )

            tmp_env = prs_ops.get_environment()
            tmp_env.update(prs_ops_env)
            prs_ops.set_environment(tmp_env)

            with prs_ops:
                pass

            for table_to_check in TABLES[search_type]:
                try:
                    features = yt_client.read_table("{}/{}".format(table_path, table_to_check))
                except Exception:
                    eh.check_failed("{} doesn't exist for {}".format(table_to_check, search_type))
                if not list(features):
                    eh.check_failed("{} is empty for {}".format(table_to_check, search_type))
