# coding=utf-8

import logging
import sys
import time

from sandbox.common import errors
from sandbox.sandboxsdk import svn
from sandbox import sdk2
from sandbox.projects.common import utils
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components.configs.prs_ops import PrsOpsCfg
from sandbox.projects.release_machine import rm_notify as rm_notify
from sandbox.projects.common.sdk_compat import task_helper
from sandbox.projects.prs_ops import resources


@rm_notify.notify2()
class RunPrsOpsNirvana(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 12 * 60 * 60
        nirvana_workflow_name = sdk2.parameters.String(
            "Workflow name",
            default_value="Sandbox driven workflow",
            description="Will be given to the newly created workflow",
            required=True
        )

        nirvana_token = sdk2.parameters.String("Vault with Nirvana Token", required=True, default_value="prs_ops_nirvana_test")
        nirvana_quota = sdk2.parameters.String("Nirvana quota", default_value="default", required=True)

        nirvana_workflow_id = sdk2.parameters.String(
            "Template Nirvana workflow id",
            description="New workflow will be copied from this one",
            required=True
        )

        nirvana_global_options = sdk2.parameters.Dict("Nirvana workflow global options")

        with sdk2.parameters.Output:
            executed_workflow_id = sdk2.parameters.String("Executed workflow id")
            executed_workflow_url = sdk2.parameters.Url("Executed workflow")

    def get_nirvana_workflow(self, workflow_id):
        sys.path.append(
            svn.Arcadia.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia/tools/mstand/nirvana_api")
        )
        import nirvana_api
        nv_token = sdk2.Vault.data(self.Parameters.nirvana_token)
        return nirvana_api.NirvanaAPI(workflow_id, nv_token)

    @staticmethod
    def get_running_blocks(n_api):
        blocks = n_api.GetWorkflow()['result']['blocks']
        block_names = dict([(block['blockCode'], block['name']) for block in blocks])

        exec_states = [dict(n_api.GetExecutionStateByCode(block['blockCode'])['result']) for block in blocks]
        block_states = [exec_state['blocks'][0] for exec_state in exec_states if len(exec_state['blocks']) == 1]
        block_statuses = [(bs['blockCode'], bs['status']) for bs in block_states if bs['status'] == 'running']

        return ["{} ({})".format(block_names[code], status) for code, status in block_statuses]

    @staticmethod
    def build_workflow_url(workflow_id):
        if workflow_id:
            return "https://nirvana.yandex-team.ru/flow/{}/graph".format(workflow_id)
        else:
            return None

    def on_enqueue(self):
        task_helper.ctx_field_set(self, rm_const.COMPONENT_CTX_KEY, PrsOpsCfg.name)

    def on_execute(self):
        template_workflow_id = self.Parameters.nirvana_workflow_id
        template_workflow_url = RunPrsOpsNirvana.build_workflow_url(template_workflow_id)
        logging.info("Template workflow: %s", template_workflow_url)

        nirvana_workflow_name = self.Parameters.nirvana_workflow_name

        n_api_orig = self.get_nirvana_workflow(template_workflow_id)
        n_api = n_api_orig.CloneThisWorkflow(nirvana_workflow_name, new_quota_project_id=self.Parameters.nirvana_quota)
        n_api.EditWorkflowSkipPermissions()
        try:
            n_api.UpdateOperationBlocks()
        except Exception as exc:
            msg = "Cannot update blocks: {} Try update source workflow: {}".format(exc, template_workflow_url)
            raise errors.TaskError(msg)

        if self.Parameters.nirvana_global_options:
            tmp = dict()
            for key, val in self.Parameters.nirvana_global_options.items():
                try:
                    tmp[key] = int(val)
                except Exception:
                    tmp[key] = val
            tmp["previous-prs-ops-resource-id"] = int(utils.last_resource_with_released_attribute(resources.PRS_OPS_EXECUTABLE).id)
            n_api.SetGlobalParametersByDict(tmp)

        n_api.StartWorkflow()

        exec_workflow_id = n_api.wfId
        exec_workflow_url = RunPrsOpsNirvana.build_workflow_url(exec_workflow_id)

        logging.info("setting executed_workflow_url = %s, executed_workflow_id = %s", exec_workflow_url, exec_workflow_id)
        self.Parameters.executed_workflow_url = exec_workflow_url
        self.Parameters.executed_workflow_id = exec_workflow_id

        message = "Starting workflow {}".format(exec_workflow_id)
        logging.info(message)

        need_to_fail = False
        while True:
            try:
                progress = dict(n_api.GetExecutionState()["result"])
                logging.info("Workflow %s progress info: %s", exec_workflow_id, progress)
                if progress["status"] == "completed":
                    execution_result = progress["result"]
                    if execution_result != "success":
                        logging.info("Workflow completed with no success (result: %s)", execution_result)
                        need_to_fail = True
                    break
            except Exception as exc:
                logging.warning("Failed to execute GetExecutionState: %s", exc)
            time.sleep(60)

        if need_to_fail:
            if execution_result == "cancel":
                raise errors.TaskFailure("Workflow has been cancelled. See {}".format(exec_workflow_url))
            else:
                raise errors.TaskFailure("Workflow has been failed. See {}".format(exec_workflow_url))

    def on_timeout(self, prev_status):
        workflow_url = self.Parameters.executed_workflow_url
        logging.info("Task timed out, workflow URL: %s, executed_workflow_id: %s", workflow_url, self.Parameters.executed_workflow_id)

        if not self.Parameters.executed_workflow_id:
            raise Exception("Task timed out, but Parameters.executed_workflow_id is empty. Could not get running blocks information")
        else:
            n_api = self.get_nirvana_workflow(self.Parameters.executed_workflow_id)
            running_blocks = self.get_running_blocks(n_api)
            if len(running_blocks) > 0:
                blocks_info = ", ".join(running_blocks)
                self.set_info(
                    "Workflow {} had timed out. Currently running blocks: {}".format(workflow_url, blocks_info),
                    do_escape=False
                )

    @property
    def footer(self):
        if self.Parameters.executed_workflow_url:
            href = str(self.Parameters.executed_workflow_url)
            return '<a href="{}">Executed workflow</a>'.format(href)
        else:
            href = RunPrsOpsNirvana.build_workflow_url(self.Parameters.nirvana_workflow_id)
            return 'Workflow not created. <a href="{}">Template workflow</a>'.format(href)
