# -*- coding: utf-8 -*-

import json
import logging

from sandbox import sdk2
from sandbox.projects.resource_types import MRCOLLECTOR_EXECUTABLE
from sandbox.projects.resource_types import YA_BLOCKSTAT_DICT
from sandbox.sdk2.helpers import subprocess


class PUMPKIN_MRCOLLECTOR_CONFIG(sdk2.Resource):
    """
    Json config for pumpkin mrcollector
    see yweb/pumpkin/mrcollector/tests/config.json
    """


class CacheUserSessions(sdk2.Task):
    """
    **Описание**
    Задача для создания кеша юзер сессий, который в дальнейшем
    используется для построения индекса Тыквы.
    """

    class Parameters(sdk2.Parameters):
        mrcollector_executable = sdk2.parameters.Resource(
            "Mrcollector executable",
            resource_type=MRCOLLECTOR_EXECUTABLE,
            required=True
        )
        mrcollector_config = sdk2.parameters.Resource(
            "Mrcollector config",
            resource_type=PUMPKIN_MRCOLLECTOR_CONFIG,
            required=True
        )
        yt_token_owner = sdk2.parameters.String("YT OAuth token owner", required=True)
        yt_token_vault = sdk2.parameters.String("YT OAuth token vault name", required=True)
        queryrec_svn_url = sdk2.parameters.ArcadiaUrl(
            "SVN url of folder with queryrec.dict and queryrec.weights",
            required=True
        )
        blockstat_dict = sdk2.parameters.Resource(
            "File blockstat.dict, use latest if not specified",
            resource_type=YA_BLOCKSTAT_DICT,
            required=False
        )

    BLOCKSTAT_DICT_CONFIG_KEY = "blockstat_dict"
    EXECUTION_DIR = "execution_dir"
    LOGGER_NAME = "cache_user_sessions"
    QUERYREC_DICT_CONFIG_KEY = "query_rec_dict"
    QUERYREC_WEIGHTS_CONFIG_KEY = "query_rec_weights"
    STAGES_CONFIG_KEY = "stages"

    def on_execute(self):
        sdk2.paths.make_folder(self.EXECUTION_DIR)

        mrcollector_executable = sdk2.ResourceData(self.Parameters.mrcollector_executable)

        mrcollector_config = sdk2.ResourceData(self.Parameters.mrcollector_config)
        mrcollector_config_contents = self._get_mrcollector_config_contents(mrcollector_config.path)
        if self.STAGES_CONFIG_KEY in mrcollector_config_contents:
            if mrcollector_config_contents[self.STAGES_CONFIG_KEY] != "1":
                logging.warning(
                    "You should not specify stages in config, only stage 1 will be launched."
                )

        self._get_queryrec_files(mrcollector_config_contents)
        self._get_blockstat_dict(mrcollector_config_contents)

        environment = {}
        yt_token = sdk2.Vault.data(
            self.Parameters.yt_token_owner,
            self.Parameters.yt_token_vault
        )
        environment.update({"YT_TOKEN": yt_token})

        call = [
            str(mrcollector_executable.path),
            "-C", str(mrcollector_config.path),
            "-T",
            "-a",
            "-s", "1",
            self.EXECUTION_DIR
        ]
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(self.LOGGER_NAME)) as pl:
            pl.logger.propagate = 1
            subprocess.Popen(
                call,
                env=environment,
                stdout=pl.stdout,
                stderr=subprocess.STDOUT
            ).wait()

    def _get_mrcollector_config_contents(self, config_path):
        mrcollector_config_contents = None
        with open(str(config_path), "r") as mrcollector_config_file:
            mrcollector_config_contents = json.load(mrcollector_config_file)
        return mrcollector_config_contents

    def _get_queryrec_files(self, config):
        queryrec_svn_url = self.Parameters.queryrec_svn_url.rstrip("/")
        queryrec_dict_file = config[self.QUERYREC_DICT_CONFIG_KEY]
        sdk2.svn.Arcadia.export(
            "{}/{}".format(queryrec_svn_url, queryrec_dict_file),
            sdk2.Path(self.EXECUTION_DIR).joinpath(queryrec_dict_file)
        )
        queryrec_weights_file = config[self.QUERYREC_WEIGHTS_CONFIG_KEY]
        sdk2.svn.Arcadia.export(
            "{}/{}".format(queryrec_svn_url, queryrec_weights_file),
            sdk2.Path(self.EXECUTION_DIR).joinpath(queryrec_weights_file)
        )

    def _get_blockstat_dict(self, config):
        blockstat_dict_resource = self.Parameters.blockstat_dict
        if not blockstat_dict_resource:
            blockstat_dict_resource = YA_BLOCKSTAT_DICT.find().first()
        blockstat_dict = sdk2.ResourceData(blockstat_dict_resource)
        sdk2.paths.copy_path(
            str(blockstat_dict.path),
            str(sdk2.Path(self.EXECUTION_DIR).joinpath(config[self.BLOCKSTAT_DICT_CONFIG_KEY])),
            symlinks=True
        )
