# -*- coding: utf-8 -*-

from sandbox import sdk2
import requests
import os
import yaml

from sandbox.projects.common import file_utils as fu
from sandbox.projects.pumpkin.CollectPumpkinPlan import PumpkinPlan
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.common.types import task as ctt


class PumpkinShooting(sdk2.Task):
    """ Task for Pumpkin shooting on st/LOAD-869 """

    class Parameters(sdk2.Task.Parameters):
        gencfg_groups = sdk2.parameters.Dict("DC / rps_schedule", default={
            "VLA_WEB_RUS_YALITE": "const(100,5m) line(1,20000,5m)",
            "MAN_WEB_RUS_YALITE": "const(100,5m) line(1,10000,5m)",
            "SAS_WEB_RUS_YALITE": "const(100,5m) line(1,5000,5m)"
        })

        pumpkin_plan = sdk2.parameters.Resource(
            "Pumpkin plan, use latest if not specified",
            resource_type=PumpkinPlan
        )

    def get_targets(self, group_name):
        target_list = []
        self.set_info('Group name: {}'.format(group_name))

        try:
            self.set_info('Asking gencfg for actual hosts in {}...'.format(group_name))
            res = requests.get('http://api.gencfg.yandex-team.ru/trunk/groups/{}'.format(group_name), timeout=15)
            target_list = res.json()['hosts']
        except Exception:
            self.set_info('No result for %s from Gencfg API', group_name)

        return target_list

    def get_tanks(self, target):
        tank_list = []
        dc = target[0:3]

        resp = requests.get('https://nanny.yandex-team.ru/v2/services/production_yandex_tank/current_state/instances/')
        nanny_result = resp.json()['result']
        for item in nanny_result:
            if item['itags'][0] == 'ALL_RCLOUD_TANKS' and item['itags'][2] == 'a_dc_'+dc:
                tank_list.append(item['container_hostname']+':'+str(item['port']))

        return tank_list

    def start_shooting(self, ammo, tanks, desc, config_content):
        subtask_shoot = ShootViaTankapi(
            self,
            ammo_source='resource',
            ammo_resource=ammo,
            description=desc,
            use_public_tanks=False,
            tanks=tanks,
            config_source='file',
            config_content=config_content,
        ).enqueue()
        self.set_info('Subtask with shooting is started')
        raise sdk2.WaitTask([subtask_shoot.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def on_execute(self):
        task_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
        config_content = fu.read_file(os.path.join(task_dir, 'PumpkinShooting/configs/', 'load.yaml'))
        const_schedule = 'const(100, 5m)'
        shoots = []

        ammo = self.Parameters.pumpkin_plan
        if ammo is None:
            ammo = PumpkinPlan.find().first()
        if ammo is None:
            raise Exception("No pumpkin plan found")

        for group, schedule in self.Parameters.gencfg_groups.items():
            targets = self.get_targets(group)
            first = True
            for target in targets:
                component = ''
                tanks = self.get_tanks(target)
                if first:
                    shoots.append({
                        'target': target.encode('utf-8'),
                        'schedule': const_schedule.encode('utf-8'),
                        'component': (group + ' [const]').encode('utf-8'),
                        'group': group.encode('utf-8'),
                        'tanks': tanks,
                        'rcassert': '',
                    })
                    component = group + ' [imbalance]'
                    first = False

                shoots.append({
                    'target': target.encode('utf-8'),
                    'schedule': schedule.encode('utf-8'),
                    'component': component.encode('utf-8'),
                    'group': group.encode('utf-8'),
                    'tanks': tanks,
                    'rcassert': '21 22 23'
                })

        for shoot in shoots:
            desc = 'Shooting target: ' + shoot['target']
            yaml_config = yaml.safe_load(config_content)
            yaml_config['phantom']['address'] = '{}:80'.format(shoot['target'])
            yaml_config['phantom']['load_profile']['schedule'] = shoot['schedule']
            yaml_config['uploader']['job_name'] = desc
            yaml_config['uploader']['component'] = shoot['component']
            yaml_config['phantom']['load_profile']['schedule'] = shoot['schedule']
            yaml_config['rcassert']['pass'] = shoot['rcassert']
            config_content = yaml.dump(yaml_config, default_flow_style=False, encoding=None)

            with self.memoize_stage["shooting_{0}_{1}".format(shoot['target'], shoot['schedule'])]:
                self.start_shooting(ammo, shoot['tanks'], desc, config_content)
