# -*- coding: utf-8 -*-

import sandbox.sdk2 as sdk2
import requests

from sandbox.sdk2 import parameters
from sandbox.common.types import resource
from sandbox.sdk2.service_resources import SandboxTasksBinary

import sandbox.projects.common.binary_task as binary_task
from sandbox.projects.qafw.selenoid import Selenoid
from sandbox.projects.qafw.webdriver_qemu import WebDriverQemu


class QafwAcceptance(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Runs tests for qafw tasks.
    """

    BROWSERS_JSON = '''
    {
      "android-phone": {
        "default": "searchapp-7.1",
        "versions": {
          "searchapp-7.1": {
            "image": "registry.yandex.net/selenium/android-phone:searchapp-7.1",
            "port": "4723",
            "path": ""
          }
        }
      },
      "chrome": {
        "default": "phone-67.0",
        "versions": {
          "pad-67.0": {
            "image": "registry.yandex.net/selenium/chrome:pad-67.0-quickboot",
            "port": "4723",
            "path": "/wd/hub"
          },
          "phone-67.0": {
            "image": "registry.yandex.net/selenium/chrome:phone-67.0-quickboot",
            "port": "4723",
            "path": "/wd/hub"
          }
        }
      }
    }'''

    class Parameters(sdk2.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        binary_resource_id = parameters.Resource(
            "Binary to test",
            resource_type=SandboxTasksBinary,
            required=True)

    class Context(sdk2.Context):
        selenoid_subtask_id = None
        webdriver_qemu_subtask_id = None

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "QAFW_SANDBOX_BINARY", "released": self.Parameters.binary_executor_release_type},
            "state": [resource.State.READY]
        }

    def _start_selenoid(self):
        subtask = Selenoid(
            self,
            binary_executor_release_type='custom',
            browsers=self.BROWSERS_JSON,
            build_id=-1,
            total=18
        )
        subtask.Requirements.tasks_resource = self.Parameters.binary_resource_id
        subtask.container = None
        subtask.save()
        subtask.enqueue()
        return subtask.id

    def _start_webdriver_qemu(self):
        subtask = WebDriverQemu(
            self,
            binary_executor_release_type='custom',
        )
        subtask.Requirements.tasks_resource = self.Parameters.binary_resource_id
        subtask.container = None
        subtask.save()
        subtask.enqueue()
        return subtask.id

    def _test_selenoid(self):
        subtask = self.find(id=self.Context.selenoid_subtask_id).first()
        assert subtask.Parameters.endpoint != "", "Selenoid was not started"
        resp = requests.get("http://{}/status".format(subtask.Parameters.endpoint))
        resp.raise_for_status()
        assert resp.json()["total"] == 18, "total browsers not equal to requested count"

    def _test_webdriver_qemu(self):
        subtask = self.find(id=self.Context.webdriver_qemu_subtask_id).first()
        assert subtask.Parameters.endpoint != "", "WebDriverQemu was not started"
        resp = requests.get("http://{}/wd/hub/status".format(subtask.Parameters.endpoint))
        resp.raise_for_status()

    def _stop_selenoid(self):
        subtask = self.find(id=self.Context.selenoid_subtask_id).first()
        subtask.stop()

    def _stop_webdriver_qemu(self):
        subtask = self.find(id=self.Context.webdriver_qemu_subtask_id).first()
        subtask.stop()

    def on_execute(self):
        with self.memoize_stage.test_step:
            self.Context.selenoid_subtask_id = self._start_selenoid()
            self.Context.webdriver_qemu_subtask_id = self._start_webdriver_qemu()
            targets = {self.Context.selenoid_subtask_id: 'endpoint',
                       self.Context.webdriver_qemu_subtask_id: 'endpoint'}
            raise sdk2.WaitOutput(targets, wait_all=True, timeout=900)

        self._test_selenoid()
        self._test_webdriver_qemu()

        self._stop_selenoid()
        self._stop_webdriver_qemu()
