# -*- coding: utf-8 -*-

import os
import re
import logging
import subprocess

import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as sb_parameters
import sandbox.sdk2.ssh as ssh
# import sandbox.projects.sandbox.resources as sb_resources


ARCANUM_KEY_NAME = 'robot_arcanum_rsa'
ARCANUM_KEY_OWNER = 'robot-arcanum'

COMMIT_RE = re.compile('^Committed revision (?P<revision>\d+)\.$', re.M)

SET_TRUST_EXP = """
#!/usr/bin/expect

set timeout 10

spawn /usr/bin/gpg --edit-key $argv 0 --yes trust quit

expect "Your decision? " { send "5\r" }
expect "Do you really want to set this key to ultimate trust? (y/N) " { send "y\r" }

interact
"""


class QafwAnsiblePlayabook(sdk2.Task):
    """
    Run ansible playbook
    """
    class Parameters(sdk2.Parameters):
        GitUrl = sb_parameters.String('Git', default='git@github.yandex-team.ru:qafw/ansible.git')
        Playbook = sb_parameters.String('Playbook path', default='')
        Hosts = sb_parameters.String('Hosts \\n-separated', default='', multiline=True)
        Tags = sb_parameters.String('Task tags comma separated', default='')
        AnsibleUser = sb_parameters.String('Ansible User')
        AnsiblePass = sb_parameters.String('Ansible Password from pass')
        _container = sdk2.parameters.Container(
            "Container", default=None, required=False,
            # type=sb_resources.LXC_CONTAINER  # FIXME: invalid argument (SANDBOX-6404)
        )

    def on_execute(self):
        git_clone_path = self._clone_ansible_git(self.Parameters.GitUrl)
        playbook = os.path.join(git_clone_path, self.Parameters.Playbook)
        if not os.path.exists(playbook):
            raise Exception("Playbook {} does not exist".format(self.Parameters.Playbook))

        inventory_path = self._make_inventory_file(self.Parameters.Hosts)
        playbook_cmd = ["ansible-playbook", playbook, "-e", "hosts=inventory", "-i", inventory_path]

        if self.Parameters.Tags:
            for tag in filter(None, self.Parameters.Tags.split(",")):
                tag = tag.strip()
                if tag:
                    playbook_cmd += ["--tags", tag]

        if self.Parameters.AnsibleUser:
            playbook_cmd += ["-u", self.Parameters.AnsibleUser]

        if self.Parameters.AnsiblePass:
            self._init_pass()
            playbook_cmd = ["pass", self.Parameters.AnsiblePass, "|"] + playbook_cmd + ["-k"]

        env = os.environ.copy()
        env["ANSIBLE_ROLES_PATH"] = os.path.join(git_clone_path, "roles")
        self._execute_command(" ".join(playbook_cmd), env=env)

    def _make_inventory_file(self, hosts):
        path = "inventory"
        inventory_content = "\n".join(
            ["[inventory]"] + filter(None, [h.strip() for h in hosts.split("\n")])
        )
        logging.info("Inventory content will is:\n%s", inventory_content)
        with open(path, "w") as f:
            f.write(inventory_content)
        return os.path.abspath(path)

    def _git_clone(self, url, path):
        self._execute_command(" ".join(["git", "clone", url, path]))

    def _clone_ansible_git(self, git_path):
        clone_path = os.path.abspath("ansible")
        self._git_clone(git_path, clone_path)
        return clone_path

    def _init_pass(self):
        pass_dir = "pass"
        os.makedirs(pass_dir)

        pub_key_path = os.path.abspath("pub.gpg")
        with open(pub_key_path, "w") as f:
            f.write(sdk2.Vault.data("QADEV", "qafw-pass-public-gpg"))

        private_key_path = os.path.abspath("private.gpg")
        with open(private_key_path, "w") as f:
            f.write(sdk2.Vault.data("QADEV", "qafw-pass-private-gpg"))
        os.chmod(private_key_path, 0400)

        set_trust_exp_path = os.path.abspath("set-trust.exp")
        with open(set_trust_exp_path, "w") as f:
            f.write(SET_TRUST_EXP)
        os.chmod(set_trust_exp_path, 0755)

        commands = [
            "gpg --import {}".format(pub_key_path),
            "gpg --allow-secret-key-import --import {}".format(private_key_path),
            # 'echo "trust\n5" | gpg --edit-key "QATools Member <qa-dev@yandex-team.ru>"',
            "/usr/bin/expect {} qa-dev@yandex-team.ru".format(set_trust_exp_path),
            'mkdir .git',
            "pass init C8843317",
            "EMAIL=robot-qafw@yandex-team.ru pass git init",
            "cd /home/sandbox/.password-store/ && git config user.email robot-qafw@yandex-team.ru && git config user.name robot-qafw && cd -",
            "pass git remote add origin ssh://git@bb.yandex-team.ru/qafw/qafw-pass.git",
            "pass git pull origin master",
        ]

        try:
            for command in commands:
                self._execute_command(command, pass_dir)
        finally:
            os.remove(pub_key_path)
            os.remove(private_key_path)

    def _execute_command(self, command, cwd=None, env=None, under_robot_qafw=True):
        def execute():
            logging.info("Running %s in %s with env %s", command, cwd, env)
            p = subprocess.Popen(command, shell=True, stderr=subprocess.PIPE, stdout=subprocess.PIPE, cwd=cwd, env=env)
            out, err = p.communicate()
            logging.info("out: %s", out)
            logging.info("err: %s", err)
            if p.returncode != 0:
                raise subprocess.CalledProcessError(p.returncode, command, out + err)
        if under_robot_qafw:
            with ssh.Key(self, "QADEV", "robot-qafw-key"):
                execute()
        else:
            execute()
