# -*- coding: utf-8 -*-

import os
import logging
import tarfile
import subprocess

import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as sb_parameters
import sandbox.sdk2.ssh as ssh
from sandbox.projects.common.utils import sync_resource


class QafwBrokenImagesMonitoringRes(sdk2.Resource):
    pass


class QafwBrokenImagesMonitoring(sdk2.Task):
    """
    Broken images monitoring https://st.yandex-team.ru/QAFW-2613
    """
    class Parameters(sdk2.Parameters):
        MonitoringPackageId = sb_parameters.String('Resource of tebroimon package')
        GridUrl = sb_parameters.String('Grid url')
        Url = sb_parameters.String('Url to check against base image')
        BaseImageId = sb_parameters.String('Base image resource id')
        Count = sb_parameters.String('Number of screenshots to be compared', default='10')

    def on_enqueue(self):
        self.Context.operation_json = QafwBrokenImagesMonitoringRes(self, 'Operation result', 'operation_res.json').id

    def on_execute(self):
        command = [
            self._get_monitoring_bin(),
            "--grid-url", self.Parameters.GridUrl,
            "--url", self.Parameters.Url,
            "--base-path", self._get_base_image(),
            "--count", self.Parameters.Count,
            "--out", os.path.abspath(str(sdk2.Resource[self.Context.operation_json].path)),
            "--send-stat",
        ]

        cwd = str(self.log_path("out"))
        os.makedirs(cwd)
        self._execute_command(command, cwd=cwd)

    def _execute_command(self, command, cwd=None, env=None, under_robot_qafw=False, shell=False):
        def execute():
            logging.info("Running %s in %s with env %s", command, cwd, env)
            p = subprocess.Popen(command, shell=shell, stderr=subprocess.PIPE, stdout=subprocess.PIPE, cwd=cwd, env=env)
            out, err = p.communicate()
            logging.info("out: %s", out)
            logging.info("err: %s", err)
            logging.info("return code: %s", p.returncode)
        if under_robot_qafw:
            with ssh.Key(self, "QADEV", "robot-qafw-key"):
                execute()
        else:
            execute()

    def _get_monitoring_bin(self):
        dest = str(self.path('package'))
        tools = str(sync_resource(self.Parameters.MonitoringPackageId))
        with tarfile.open(tools) as tf:
            tf.extractall(dest)
        return os.path.join(dest, "bin", "tebroimon")

    def _get_base_image(self):
        return str(sync_resource(self.Parameters.BaseImageId))
