import json
import math
import logging
from sandbox.projects.qafw.common.sandboxrouter import SandboxRouterComponent


class BrowserGroupLauncher(object):
    def __init__(self, group_config, token, timeout, host, path, cwd):
        self.group_config = group_config
        self.token = token
        self.timeout = timeout
        self.host = host
        self.path = path
        self._cwd = cwd
        self._config_path = self._cwd.joinpath("{}.config.json".format(self.group_config.name))
        self._routers = self._init_routers()

    def browsers(self):
        return self.group_config.json()["parameters"]["browsers"]

    @property
    def routers(self):
        return self._routers

    def _get_child_count(self):
        return int(math.ceil(float(self.group_config.total) / float(self.group_config.capacity)))

    def _init_routers(self):
        routers = []
        count = self._get_child_count()
        for index in xrange(count):
            router = SandboxRouterComponent(self.path,
                                            config=str(self._config_path),
                                            token=self.token,
                                            start_timeout=self.timeout,
                                            host=self.host)
            routers.append(router)
        return routers

    def start(self):
        self._dump_config()
        for router in self.routers:
            router.start()

    def stop(self):
        for router in self.routers:
            router.stop()

    def wait(self):
        for router in self.routers:
            router.wait()

    def _dump_config(self):
        with self._config_path.open("w") as f:
            f.write(unicode(json.dumps(self.group_config.json())))
        logging.info("dump: {}".format(unicode(json.dumps(self.group_config.json()))))
        return str(self._config_path)


class BrowserGroup(object):
    def __init__(self, task, name, timeout, total, capacity, owner, requirements, priority, **kwargs):
        self._kwargs = kwargs
        self._total = total
        self._name = name
        self._capacity = capacity
        self._task = task
        self._timeout = timeout
        self._owner = owner
        self._requirements = requirements
        self._priority = priority

    @property
    def name(self):
        return self._name

    @property
    def task(self):
        return self._task

    @property
    def timeout(self):
        return self._timeout

    @property
    def total(self):
        return self._total

    @property
    def capacity(self):
        return self._capacity

    @property
    def requirements(self):
        return {
            "numCores": self._requirements["ncpu"],
            "ram": self._requirements["ram"],
            "disk": self._requirements["disk"],
        }

    def json(self):
        result = {"parameters": self._kwargs,
                  "type": self._task,
                  "quota": self._owner,
                  "timeout": "{}s".format(self._timeout),
                  "requirements": self.requirements,
                  "priority": self._priority}
        result["parameters"]["total"] = self._capacity
        return result
