import os.path
import time
import logging
import sandbox.projects.common.network as network
from sandbox.projects.common.search.components import component
import xml.etree.ElementTree as xml


class GoGridRouterConfigurableComponent(component.ProcessComponentMixin,
                                        component.WaitUrlComponentMixin,
                                        component.Component):
    """Go Grid Router component"""

    def __init__(self, path, cwd, start_timeout=60, config=None, **kwargs):
        self._port = component.try_get_free_port()
        self._host = network.get_my_ipv6()
        self._cwd = cwd
        self._config = GoGridRouterConfig() if config is None else config
        self.__name = os.path.basename(path)
        args = [path,
                "-listen", ":{}".format(self.port),
                "-quotaDir", str(self.quota_dir),
                "-guests-quota", "sandbox",
                "-guests-allowed"]

        logging.info("Using port {} for {}".format(self._port, self.__name))

        component.ProcessComponentMixin.__init__(self, args, **kwargs)

        component.WaitUrlComponentMixin.__init__(
            self,
            url="http://{}:{}/ping".format("localhost", self._port),
            wait_timeout=start_timeout
        )

    @property
    def port(self):
        return int(self._port)

    @property
    def config(self):
        return self._config

    @property
    def host(self):
        return "[{}]".format(str(self._host))

    def wait_while_running(self):
        while self.process.poll() is None:
            time.sleep(1)

    @property
    def quota_dir(self):
        return self._cwd

    def start(self):
        self._config.dump(self.quota_dir.joinpath("sandbox.xml"))
        component.ProcessComponentMixin.start(self)


class GoGridRouterComponent(GoGridRouterConfigurableComponent):
    """Go Grid Router component"""

    def __init__(self, path, cwd, start_timeout=60, **kwargs):
        super(GoGridRouterComponent, self).__init__(path, cwd, start_timeout, **kwargs)
        self._launchers = []

    def _build_config(self):
        for launcher in self._launchers:
            group = GoGridRouterGroupItem()

            browsers = launcher.browsers()
            routers = launcher.routers
            capacity = launcher.group_config.capacity

            for router in routers:
                host_item = GoGridRouterHostItem(router.host, router.port, capacity)
                group.endpoints.append(host_item)

            for browser in browsers:
                browser_item = GoGridRouterBrowserItem(browser)
                for version in browsers[browser]["versions"]:
                    browser_item.versions.append(version)
                group.browsers.append(browser_item)

            self.config.groups.append(group)

    def add_launcher(self, launcher):
        self._launchers.append(launcher)

    def start(self):
        for launcher in self._launchers:
            launcher.start()
        self._build_config()
        self._config.dump(self._cwd.joinpath("sandbox.xml"))
        component.ProcessComponentMixin.start(self)

    def stop(self):
        for launcher in self._launchers:
            launcher.stop()
        component.ProcessComponentMixin.stop(self)

    def wait(self):
        for launcher in self._launchers:
            launcher.wait()
        component.WaitUrlComponentMixin.wait(self)


class GoGridRouterConfig(object):
    """Go Grid Router config"""

    def __init__(self, groups=None):
        self._groups = [] if groups is None else groups

    @property
    def groups(self):
        return self._groups

    def _dump_group(self, group):
        bro_el_dict = {}
        for browser in group.browsers:
            bro_el_dict[browser.name] = xml.Element('browser')
            bro_el_dict[browser.name].attrib["name"] = browser.name
            default_version = None
            for version in browser.versions:
                default_version = version
                ver_el = xml.Element('version')
                ver_el.attrib["number"] = version

                reg_el = xml.Element('region')
                reg_el.attrib["name"] = "sandbox"
                for host_item in group.endpoints:
                    host_elem = xml.Element('host')
                    host_elem.attrib["name"] = host_item.host_name
                    host_elem.attrib["port"] = str(host_item.port)
                    host_elem.attrib["count"] = str(host_item.count)
                    reg_el.append(host_elem)
                ver_el.append(reg_el)
                bro_el_dict[browser.name].append(ver_el)
            bro_el_dict[browser.name].attrib["defaultVersion"] = default_version
        return bro_el_dict.values()

    def dump(self, config_filename):
        xml.register_namespace('qa', "urn:config.gridrouter.qatools.ru")
        config = xml.Element("{urn:config.gridrouter.qatools.ru}browsers")
        for group in self._groups:
            config.extend(self._dump_group(group))

        config_filename = config_filename
        config_filename.open("w").write(unicode(xml.tostring(config)))


class GoGridRouterGroupItem(object):
    """Go Grid Router config GroupItem"""

    def __init__(self, browsers=None, endpoints=None):
        self._browsers = [] if browsers is None else browsers
        self._endpoints = [] if endpoints is None else endpoints

    @property
    def browsers(self):
        return self._browsers

    @property
    def endpoints(self):
        return self._endpoints


class GoGridRouterHostItem(object):
    """Go Grid Router config HostItem"""

    def __init__(self, host_name, port, count):
        self._host_name = host_name
        self._port = port
        self._count = count

    @property
    def host_name(self):
        return self._host_name

    @property
    def port(self):
        return self._port

    @property
    def count(self):
        return self._count


class GoGridRouterBrowserItem(object):
    """Go Grid Router config BrowserItem"""

    def __init__(self, name, versions=None):
        self._name = name
        self._versions = [] if versions is None else versions

    @property
    def name(self):
        return self._name

    @property
    def versions(self):
        return self._versions
