import os.path
import time
import socket
import logging
import sandbox.projects.common.network as network
from sandbox.projects.common.search.components import component
from sandbox.projects.qafw.common import socatlauncher


class QemuLauncher(component.ProcessComponentMixin,
                   component.WaitPortComponentMixin,
                   component.Component):
    """QEMU component
    xvfb-run -l -n $DISPLAY_NUM -s "-ac -screen 0 $SCREEN_RESOLUTION -noreset -listen tcp"
    qemu-system-x86_64 -enable-kvm -machine q35,accel=kvm,usb=off -smp sockets=1,cores=1,
    threads=2 -m 2048 -boot strict=on -usbdevice tablet -net nic,vlan=0 -net user,vlan=0,hostfwd=tcp::4444-:4444
    -drive file=image.img,media=disk,if=virtio,snapshot=on
    """
    QEMU_BIN = "qemu-system-x86_64"
    XVFB_PATH = "xvfb-run"

    def __init__(self, image, start_timeout=600, vm_cores=5, vm_ram=8, display=99, processes=1, **kwargs):
        self._image = image
        self._display = display
        self._ip = network.get_my_ipv6()
        self._host = "[{}]".format(str(self._ip))

        self._endpoints = [(self._ip, component.try_get_free_port()) for _ in xrange(processes)]
        self._socats = [socatlauncher.SocatLauncher(endpoint[1]) for endpoint in self._endpoints]

        self.__name = os.path.basename(QemuLauncher.QEMU_BIN)
        hostfwd_str = ','.join(
            ['hostfwd=tcp::{}-:{}'.format(endpoint[1], 4444+self._endpoints.index(endpoint))
             for endpoint in self._endpoints])
        args = [QemuLauncher.XVFB_PATH,
                "-l",
                "-n", "{}".format(self._display),
                "-s", "-ac -screen 0 1600x1200x24 -noreset -listen tcp",
                QemuLauncher.QEMU_BIN,
                "-enable-kvm",
                "-machine", "pc,accel=kvm,usb=off",
                "-smp", "{cores},sockets=1,cores={cores},threads=1".format(cores=vm_cores),
                "-m", "{mem}G".format(mem=vm_ram),
                "-boot", "strict=on",
                "-device", "e1000,netdev=nw0",
                "-netdev", "user,id=nw0,ipv6-net=2001:db8:0:2::/64,{}".format(hostfwd_str),
                "-device", "VGA,id=vga0,vgamem_mb=128",
                "-drive", "file={image},media=disk,if=virtio,snapshot=on,cache=unsafe".format(image=self._image),
                "-no-hpet", "-global", "kvm-pit.lost_tick_policy=discard", "-rtc", "driftfix=slew"
                ]
        logging.info("Using ports {} for {}".format([endpoint[1] for endpoint in self._endpoints], self.__name))

        component.ProcessComponentMixin.__init__(self, args, **kwargs)

        component.WaitPortComponentMixin.__init__(
            self,
            endpoints=self._endpoints,
            wait_timeout=start_timeout
        )

    def start(self):
        for socat in self._socats:
            socat.start()
        component.ProcessComponentMixin.start(self)

    def stop(self):
        """stop socat before"""
        for socat in self._socats:
            socat.stop()
        component.ProcessComponentMixin.stop(self)

    def wait(self):
        """Ignoring socket errors"""
        for socat in self._socats:
            socat.wait()
        while True:
            try:
                component.WaitPortComponentMixin.wait(self)
                break
            except socket.error:
                time.sleep(2)
                continue

    @property
    def endpoints(self):
        return self._endpoints

    @property
    def host(self):
        return self._host

    def wait_while_running(self):
        self.process.wait()
