# -*- coding: utf-8 -*-

import os
import tarfile
import logging
import datetime
import subprocess

import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as sb_parameters
import sandbox.sdk2.ssh as ssh

from sandbox.projects.common.utils import sync_resource


class SeleniumGridDailyUsage(sdk2.Resource):
    pass


class QafwCalcQuotaDailyUsage(sdk2.Task):
    """
    Calculate quota daily usage
    """
    class Parameters(sdk2.Parameters):
        CalcGridUsagePackageId = sb_parameters.String('Resource of grid_usage package')
        QuotaGit = sb_parameters.String('Git', default='ssh://git@bb.yandex-team.ru/selenium/grid-router-deb.git')
        Date = sb_parameters.String('Date to calc usage for usage calculation (%Y%m%d)', default='Today')
        Groups = sb_parameters.String('Groups \\n-separated', default='', multiline=True)

    def on_enqueue(self):
        self.Context.metrics_json = SeleniumGridDailyUsage(self, 'Selenium grid daily usage for {}'.format(self.Parameters.Date), 'mertrics.json').id

    def on_execute(self):
        if self.Parameters.Date.lower().strip() == "today":
            d = datetime.datetime.now()
        else:
            d = datetime.datetime.strptime(self.Parameters.Date, '%Y%m%d')

        ggr_deb_path = self._clone_quota_git(self.Parameters.QuotaGit)
        self._execute_command('git checkout "`git rev-list master  -n 1 --first-parent --before={date}`"'.format(date=d.strftime('%Y-%m-%d')), cwd=ggr_deb_path)

        quota_path = os.path.join(ggr_deb_path, "quota.json")
        assert os.path.exists(quota_path), "quota {} does not exist".format(quota_path)

        calc_usage_bin = self._get_calc_usage_bin()
        self._execute_command(
            " ".join([
                calc_usage_bin,
                "--quota", quota_path,
                "--date", d.strftime('%Y%m%d'),
                "--out", str(sdk2.Resource[self.Context.metrics_json].path),
            ] + ["--host-group={}".format(hg) for hg in filter(None, [g.strip() for g in self.Parameters.Groups.split("\n")])])
        )

    def _git_clone(self, url, path):
        self._execute_command(" ".join(["git", "clone", url, path]))

    def _clone_quota_git(self, git_path):
        clone_path = os.path.abspath("grid-router-deb")
        self._git_clone(git_path, clone_path)
        return clone_path

    def _get_calc_usage_bin(self):
        dest = str(self.path('rtc'))
        tools = str(sync_resource(self.Parameters.CalcGridUsagePackageId))
        with tarfile.open(tools) as tf:
            tf.extractall(dest)
        return os.path.join(dest, "bin", "calc_usage")

    def _execute_command(self, command, cwd=None, env=None, under_robot_qafw=True):
        def execute():
            logging.info("Running %s in %s with env %s", command, cwd, env)
            p = subprocess.Popen(command, shell=True, cwd=cwd, env=env)
            p.communicate()
            if p.returncode != 0:
                raise subprocess.CalledProcessError(p.returncode, command)
        if under_robot_qafw:
            with ssh.Key(self, "QADEV", "robot-qafw-key"):
                execute()
        else:
            execute()
