# -*- coding: utf-8 -*-

import json
import logging


from sandbox.common.types import task as ctt
from sandbox.common.rest import Client
from sandbox.common.types import resource
from sandbox.projects.qafw.resource_types import SeleniumQemuImage
from sandbox.projects.qafw.resource_types import SeleniumGridRouter
from sandbox.projects.qafw.resource_types import SeleniumGgrUi
from sandbox.projects.qafw.resource_types import SeleniumSandboxRouter
from sandbox.projects.qafw.resource_types import SeleniumPlatformConfig
from sandbox.projects.qafw.common.browsergroup import BrowserGroup
from sandbox.projects.qafw.common.browsergroup import BrowserGroupLauncher
from sandbox.projects.qafw.common.gogridrouter import GoGridRouterComponent
from sandbox.projects.qafw.common.ggr_ui import GgrUiComponent

import sandbox.sdk2 as sdk2
import sandbox.projects.common.binary_task as binary_task


def _load_browsers_json(platform_resource, config_type):
    browsers_file = \
        str(sdk2.ResourceData(
            platform_resource).path / '{config_type}.json'.format(
            config_type=config_type))
    browsers = json.load(open(browsers_file))
    return browsers


def _get_browser_config_from_requirements(browser_name, browser_version, browser_requirements):
    for requirement in browser_requirements:
        if browser_name == requirement["name"] and browser_version == requirement["version"]:
            return requirement
    return None


def _is_browser_required(browser_name, browser_version, browser_requirements):
    return _get_browser_config_from_requirements(browser_name, browser_version, browser_requirements) is not None


def _get_browser_total(browser_name, browser_version, browser_requirements):
    return _get_browser_config_from_requirements(browser_name, browser_version, browser_requirements).get("total", 1)


def _get_browser_requests(browser_name, browser_version, browser_requirements):
    browser_config = _get_browser_config_from_requirements(browser_name, browser_version, browser_requirements)
    return browser_config.get("total", 1), browser_config.get("cpu_ratio", 0)


def _build_browser_config(platform_resource, config_type, browser_requirements):
    browsers = _load_browsers_json(platform_resource, config_type)
    config = {}
    total = 0
    cpu_ratio = 0
    for browser_name, available_versions in browsers.items():
        versions = {}
        for browser_version, version_config in available_versions["versions"].items():
            if _is_browser_required(browser_name, browser_version, browser_requirements):
                versions[browser_version] = version_config
                browser_total, cpu_ratio = _get_browser_requests(browser_name, browser_version, browser_requirements)
                total += browser_total
        if len(versions) > 0:
            config[browser_name] = {"versions": versions, "default": versions.keys()[0]}
    return config, total, cpu_ratio


def _create_docker_browsers_config(platform_resource, platform_name, platforms, browser_requirements):
    config = platforms[platform_name]
    config.setdefault("total", 0)
    browsers_json, total, cpu_ratio = _build_browser_config(platform_resource, platform_name, browser_requirements)
    config["requirements"] = browsers_json
    config["total"] = total
    if cpu_ratio > 0:
        config["cpu_ratio"] = cpu_ratio
    return config


def _create_qemu_browsers_config(platform_name, platforms, browser_requirements):
    config = platforms[platform_name]
    config.setdefault("total", 0)
    browser_name, browser_version = platform_name.split(":")
    if _is_browser_required(browser_name, browser_version, browser_requirements):
        config["requirements"] = {browser_name: {"versions": {browser_version: {}}}}
        browser_requirements = _get_browser_config_from_requirements(browser_name, browser_version, browser_requirements)
        if "image" in browser_requirements:
            config["image"] = browser_requirements["image"]
        elif "image" in platforms:
            config["image"] = platforms["image"]
        else:
            resource = SeleniumQemuImage.find(attrs={
                "browser": browser_name,
                "version": browser_version,
                "released": "stable"
            }).first()
            config["image"] = resource.id

        config.setdefault("count", browser_requirements.get("count", 1))
        config["total"] = _get_browser_total(browser_name, browser_version, browser_requirements)
    return config


def _create_platforms_config(platform_resource, browser_requirements):
    config = {}
    with open(str(sdk2.ResourceData(platform_resource).path / "platform.json")) as platforms_config:
        platforms = json.load(platforms_config)
        for platform_name in platforms:
            if ":" not in platform_name:
                config[platform_name] = _create_docker_browsers_config(platform_resource, platform_name, platforms, browser_requirements)
            else:
                config[platform_name] = _create_qemu_browsers_config(platform_name, platforms, browser_requirements)
    return config


class Ggr(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Runs GGR task
    """

    LINUX = "linux"
    ANDROID = "android"
    QEMU = "qemu"

    class Requirements(sdk2.Requirements):
        cores = 8
        disk_space = 1 * 1024  # 30 GB
        ram = 8 * 1024  # 16 GB

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Project build'):
            build_id = sdk2.parameters.Integer(
                'Build ID',
                hint=True,
                default=0,
            )
        with sdk2.parameters.Group("Common"):
            start_timeout = sdk2.parameters.Integer("Timeout for subtask starting", default=300)

        with sdk2.parameters.Group("Execution"):
            browsers = sdk2.parameters.JSON("Execution requirements", multiline=True, default=[])

        with sdk2.parameters.Group("Selenium Resources"):
            platform = sdk2.parameters.LastReleasedResource(
                "Platform configuration",
                description="Platform configuration. Images, hardware requirements, capacity, etc",
                resource_type=SeleniumPlatformConfig,
                required=False
            )

            ggr = sdk2.parameters.LastReleasedResource(
                "GGR resource",
                description="Resource to use in task",
                resource_type=SeleniumGridRouter,
                required=False
            )

            ggr_ui = sdk2.parameters.LastReleasedResource(
                "GGR UI resource",
                description="Resource to build charts",
                resource_type=SeleniumGgrUi,
                required=False
            )

            sbr = sdk2.parameters.LastReleasedResource(
                "Sandbox router resource",
                description="Resource to use in task",
                resource_type=SeleniumSandboxRouter,
                required=False
            )

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Output():
            endpoint = sdk2.parameters.String(
                "GGR address",
            )
            status_endpoint = sdk2.parameters.String(
                "/status API address",
            )

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "QAFW_SANDBOX_BINARY", "released": self.Parameters.binary_executor_release_type},
            "state": [resource.State.READY]
        }

    def _sbr_bin(self):
        return str(sdk2.ResourceData(self.Parameters.sbr).path)

    def _ggr_bin(self):
        return str(sdk2.ResourceData(self.Parameters.ggr).path)

    def _ggr_ui_bin(self):
        return str(sdk2.ResourceData(self.Parameters.ggr_ui).path)

    def _create_group_launcher(self, task, name, total, capacity, requirements, **kwargs):
        priority = ctt.Priority.as_dict(self.Parameters.priority)
        group_config = BrowserGroup(task, name, self.Parameters.kill_timeout, total, capacity, self.owner, requirements, priority, **kwargs)
        launcher = BrowserGroupLauncher(
            group_config,
            Client._external_auth.task_token,
            self.Parameters.start_timeout,
            self.server.host,
            self._sbr_bin(),
            cwd=self.path())
        return launcher

    def on_execute(self):
        self._go_grid_router = None
        self._ggr_ui = None
        try:
            self._go_grid_router = GoGridRouterComponent(self._ggr_bin(), cwd=self.path())

            platforms = _create_platforms_config(self.Parameters.platform, self.Parameters.browsers)
            logging.debug("Platforms config: {}".format(platforms))
            for platform_name in platforms:
                platform_config = platforms[platform_name].copy()

                # Need to remove the following keys remapping
                if "requirements" in platform_config:
                    platform_config["browsers"] = platform_config.pop("requirements")
                platform_config["requirements"] = platform_config.pop("sandbox_host_requirements")

                if (
                        platform_config.get("capacity") and
                        (platform_config.get("total") or platform_config.get("cpu_ratio"))
                ):
                    launcher = self._create_group_launcher(name=platform_name,
                                                           build_id=self.Parameters.build_id,
                                                           **platform_config)
                    self._go_grid_router.add_launcher(launcher)

            self._go_grid_router.start()
            self._go_grid_router.wait()

            self._ggr_ui = GgrUiComponent(self._ggr_ui_bin(), cwd=self.path(), ggr=self._go_grid_router)
            self._ggr_ui.start()
            self._ggr_ui.wait()

            self.Parameters.status_endpoint = "{}:{}".format(self._ggr_ui.host, self._ggr_ui.port)
            self.Parameters.endpoint = "{}:{}".format(self._go_grid_router.host, self._go_grid_router.port)
            self.set_info("Webdriver endpoint = {}, status endpoint = {}".format(self.Parameters.endpoint, self.Parameters.status_endpoint))

            self._go_grid_router.wait_while_running()
        finally:
            self._stop()

    def _stop(self):
        if self._go_grid_router:
            self._go_grid_router.stop()
        if self._ggr_ui:
            self._ggr_ui.stop()

    def on_terminate(self):
        self._stop()

    def on_before_timeout(self, seconds):
        self._stop()
