# -*- coding: utf-8 -*-

import os
import logging
import tarfile
import subprocess

import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as sb_parameters
import sandbox.sdk2.ssh as ssh
from sandbox.projects.common.utils import sync_resource


class SeleniumQypOperationRes(sdk2.Resource):
    pass


class QafwQypKicker(sdk2.Task):
    """
    Tries to fix windows machine states in qyp
    """
    class Parameters(sdk2.Parameters):
        QypDeployPackageId = sb_parameters.String('Resource of qyp package', default='740285906')
        QuotaGit = sb_parameters.String('Git', default='ssh://git@bb.yandex-team.ru/selenium/grid-router-deb.git')
        IncludedGroups = sb_parameters.String('Included host groups \\n-separated', default='', multiline=True)
        RevertInvalid = sb_parameters.Bool('Revert invalid machines', default=True)

    def on_execute(self):
        ggr_deb_path = self._clone_quota_git(self.Parameters.QuotaGit)
        quota_path = os.path.join(ggr_deb_path, "quota.json")
        assert os.path.exists(quota_path), "quota {} does not exist".format(quota_path)

        ya_path = str(self.path("ya"))
        sdk2.svn.Arcadia.export(
            "arcadia:/arc/trunk/arcadia/ya",
            ya_path,
        )

        coinfig_path = str(self.path("qyp.yaml"))
        sdk2.svn.Arcadia.export(
            "arcadia:/arc/trunk/arcadia/devtools/qafw/qyp/qyp.yaml",
            coinfig_path,
        )

        env = os.environ.copy()
        env["QYP_TOKEN"] = sdk2.Vault.data("QADEV", "qafw_qyp_token")

        command = [
            self._get_qyp_bin(),
            "--ya", ya_path,
            "--quota", quota_path,
            "--config", coinfig_path,
        ]

        for i in filter(None, [g.strip() for g in self.Parameters.IncludedGroups.split("\n")]):
            command += ["--group", i]

        if self.Parameters.RevertInvalid:
            command += ["--fix-revert"]
        else:
            command += ["--fix"]

        rc = self._execute_command(
            " ".join(command),
            env=env,
        )
        assert rc == 0

    def _git_clone(self, url, path):
        self._execute_command(" ".join(["git", "clone", url, path]))

    def _clone_quota_git(self, git_path):
        clone_path = os.path.abspath("grid-router-deb")
        self._git_clone(git_path, clone_path)
        return clone_path

    def _execute_command(self, command, cwd=None, env=None, under_robot_qafw=True):
        def execute():
            logging.info("Running %s in %s with env %s", command, cwd, env)
            p = subprocess.Popen(command, shell=True, stderr=subprocess.PIPE, stdout=subprocess.PIPE, cwd=cwd, env=env)
            out, err = p.communicate()
            logging.info("out: %s", out)
            logging.info("err: %s", err)
            logging.info("return code: %s", p.returncode)
            return p.returncode

        if under_robot_qafw:
            with ssh.Key(self, "QADEV", "robot-qafw-key"):
                return execute()
        else:
            return execute()

    def _get_qyp_bin(self):
        dest = str(self.path('qyp_package'))
        tools = str(sync_resource(self.Parameters.QypDeployPackageId))
        with tarfile.open(tools) as tf:
            tf.extractall(dest)
        return os.path.join(dest, "bin", "qyp")
