# -*- coding: utf-8 -*-

import os
import logging
import tarfile
import subprocess

import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as sb_parameters
import sandbox.sdk2.ssh as ssh
from sandbox.projects.common.utils import sync_resource


class SeleniumRtcOperationRes(sdk2.Resource):
    pass


class QafwRtcKicker(sdk2.Task):
    """
    Tries to fix windows machine states in RTC: starts stopped, reverts and starts invalid ones
    """
    class Parameters(sdk2.Parameters):
        RtcDeployPackageId = sb_parameters.String('Resource of rtc_deploy package', default='740285906')
        QuotaGit = sb_parameters.String('Git', default='ssh://git@bb.yandex-team.ru/selenium/grid-router-deb.git')
        IncludedGroups = sb_parameters.String('Included host groups \\n-separated', default='', multiline=True)
        ExcludedGroups = sb_parameters.String('Included host groups \\n-separated', default='rtc-bad', multiline=True)
        RevertInvalid = sb_parameters.Bool('Revert invalid machines', default=True)

    def on_enqueue(self):
        self.Context.operation_json = SeleniumRtcOperationRes(self, 'Operation result', 'operation_res.json').id

    def on_execute(self):
        ggr_deb_path = self._clone_quota_git(self.Parameters.QuotaGit)
        quota_path = os.path.join(ggr_deb_path, "quota.json")
        assert os.path.exists(quota_path), "quota {} does not exist".format(quota_path)

        sdk2.svn.Arcadia.export(
            "arcadia:/arc/trunk/arcadia/devtools/qafw/rtc_deploy/config.json",
            "config.json",
        )
        calc_usage_script = "config.json"

        env = os.environ.copy()
        env["NANNY_TOKEN"] = sdk2.Vault.data("QADEV", "qafw_nanny_token")
        env["RTC_TOKEN"] = sdk2.Vault.data("QADEV", "qafw_rtc_token")

        base_command = [
            self._get_rtc_bin(),
            "-q", quota_path,
            "-c", calc_usage_script,
            "--out-json", str(sdk2.Resource[self.Context.operation_json].path),
        ]

        for e in filter(None, [g.strip() for g in self.Parameters.ExcludedGroups.split("\n")]):
            base_command += ["-e", e]

        for i in filter(None, [g.strip() for g in self.Parameters.IncludedGroups.split("\n")]):
            base_command += ["-i", i]

        if self.Parameters.RevertInvalid:
            self._execute_command(
                " ".join(base_command + ["--revert-invalid"]),
                env=env,
            )

        self._execute_command(
            " ".join(base_command + ["--start"]),
            env=env,
        )

    def _git_clone(self, url, path):
        self._execute_command(" ".join(["git", "clone", url, path]))

    def _clone_quota_git(self, git_path):
        clone_path = os.path.abspath("grid-router-deb")
        self._git_clone(git_path, clone_path)
        return clone_path

    def _execute_command(self, command, cwd=None, env=None, under_robot_qafw=True):
        def execute():
            logging.info("Running %s in %s with env %s", command, cwd, env)
            p = subprocess.Popen(command, shell=True, stderr=subprocess.PIPE, stdout=subprocess.PIPE, cwd=cwd, env=env)
            out, err = p.communicate()
            logging.info("out: %s", out)
            logging.info("err: %s", err)
            logging.info("return code: %s", p.returncode)
        if under_robot_qafw:
            with ssh.Key(self, "QADEV", "robot-qafw-key"):
                execute()
        else:
            execute()

    def _get_rtc_bin(self):
        dest = str(self.path('rtc'))
        tools = str(sync_resource(self.Parameters.RtcDeployPackageId))
        with tarfile.open(tools) as tf:
            tf.extractall(dest)
        return os.path.join(dest, "bin", "rtc")
