# -*- coding: utf-8 -*-

import os
import json
import time
import logging
import subprocess

import sandbox.sdk2.ssh as ssh
import sandbox.sdk2 as sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.projects.sandbox.resources as sb_resources
from sandbox.common.types import resource
import sandbox.projects.common.network as network
import sandbox.projects.common.binary_task as binary_task


class Selenoid_Beta(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Runs beta selenoid task. For testing purpose only.
    """

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.SSD

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        container = sdk2.parameters.Container(
            "Environment container resource",
            default_value="980356616",
            resource_type=sb_resources.LXC_CONTAINER,
            platform="linux_ubuntu_16.04_xenial",
            required=True,
        )
        with sdk2.parameters.Group('Project build') as project_build_block:
            build_id = sdk2.parameters.Integer(
                'Build ID',
                hint=True,
                default=0,
            )
        browsers = sdk2.parameters.String('Browsers json', default='', multiline=True)
        total = sdk2.parameters.Integer("Selenoid's Total", default=50)
        session_attempt_timeout = sdk2.parameters.String("Selenoid's session attempt timeout", default='120s')
        service_startup_timeout = sdk2.parameters.String("Selenoid's service startup timeout", default='120s')
        use_robot_qafw = sdk2.parameters.Bool("Use QAFW robot", default=True)

        with sdk2.parameters.Output():
            endpoint = sdk2.parameters.String(
                "selenoid address",
            )

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "QAFW_SANDBOX_BINARY", "released": self.Parameters.binary_executor_release_type},
            "state": [resource.State.READY]
        }

    def _get_s3(self):
        try:
            return {
                "endpoint": "https://s3.mds.yandex.net",
                "region": "us-west-1",
                "bucket": "sessions",
                "access_key": sdk2.Vault.data("QADEV", "selenoid_s3_access_key"),
                "secret_key": sdk2.Vault.data("QADEV", "selenoid_s3_secret_key"),
            }
        except Exception, e:
            logging.debug("no acces to s3 credentials: {}".format(e.message))
            return None

    def on_execute(self):
        browsers = json.loads(self.Parameters.browsers)

        self._execute_docker("docker pull registry.yandex.net/selenium/selenoid:1.8.4-s3")
        self._execute_docker("docker pull registry.yandex.net/selenium/video-recorder:latest-release")

        etc = str(self.path("etc"))
        logs = str(self.log_path("logs"))
        video = str(self.log_path("video"))
        for d in [etc, os.path.join(etc, "selenoid"), logs, video]:
            os.makedirs(d)

        with open(os.path.join(etc, "selenoid", "browsers.json"), "w") as f:
            json.dump(browsers, f)

        self._download_browser_images(browsers)

        exec_arguments = [
            "docker run -d",
            "--name selenoid",
            "--restart always",
            "-e TZ=Europe/Moscow",
            "-e OVERRIDE_VIDEO_OUTPUT_DIR={video}",
            "-p 4444:4444",
            "--log-opt max-size=500m",
            "--log-opt max-file=7",
            "-v {etc}/selenoid:/etc/selenoid:ro",
            "-v {video}:/opt/selenoid/video/",
            "-v {logs}:/opt/selenoid/logs/",
            "-v /var/run/docker.sock:/var/run/docker.sock",
            "registry.yandex.net/selenium/selenoid:1.8.4-s3 -limit {}".format(self.Parameters.total),
            "-conf /etc/selenoid/browsers.json",
            "-video-output-dir /opt/selenoid/video",
            "-video-recorder-image registry.yandex.net/selenium/video-recorder:latest-release",
            "-log-output-dir /opt/selenoid/logs",
            "-session-attempt-timeout {}".format(self.Parameters.session_attempt_timeout),
            "-service-startup-timeout {}".format(self.Parameters.service_startup_timeout),
        ]
        s3 = self._get_s3()
        if s3:
            logging.info("Use s3 to store sessions")
            exec_arguments.extend([
                "-s3-endpoint '{}'".format(s3["endpoint"]),
                "-s3-region '{}'".format(s3["region"]),
                "-s3-bucket-name '{}'".format(s3["bucket"]),
                "-s3-access-key '{}'".format(s3["access_key"]),
                "-s3-secret-key '{}'".format(s3["secret_key"]),
                "-s3-exclude-files '*.json'",
                "-s3-keep-files"
            ])

        self._execute_docker(
            " ".join(exec_arguments).format(etc=etc, video=video, logs=logs)
        )

        addr = network.get_my_ipv6()
        self.set_info("IP: {}".format(addr))
        self.Parameters.endpoint = "[{}]:4444".format(addr)

        while True:
            time.sleep(1)

    def on_finish(self, prev_status, status):
        self._stop_selenoid()

    # def on_break(self, prev_status, status):
    #     self._stop_selenoid()

    def _stop_selenoid(self):
        self._execute_docker("docker stop selenoid")

    def _create_single_bash_command(self, command_list):
        command_list.append("wait")
        return "{}".format(" & ".join(command_list))

    def _download_browser_images(self, browsers):
        commands = []
        for browser in browsers:
            for version in browsers[browser]["versions"]:
                if "image" in browsers[browser]["versions"][version]:
                    image = browsers[browser]["versions"][version]["image"]
                    commands.append("docker pull {}".format(image))
        self._execute_docker(self._create_single_bash_command(commands))

    def _execute_docker(self, command, cwd=None, env=None):
        under_robot_qafw = bool(self.Parameters.use_robot_qafw)
        env = env or os.environ.copy()
        env["DOCKER_CONFIG"] = "/etc/docker/.docker"
        return self._execute_command(command=command, cwd=cwd, env=env, under_robot_qafw=under_robot_qafw)

    def _execute_command(self, command, cwd=None, env=None, under_robot_qafw=True):
        def execute():
            logging.info("Running %s in %s with env %s", command, cwd, env)
            p = subprocess.Popen(command, shell=True, cwd=cwd, env=env)
            p.communicate()
            if p.returncode != 0:
                raise subprocess.CalledProcessError(p.returncode, command)

        if under_robot_qafw:
            with ssh.Key(self, "QADEV", "robot-qafw-key"):
                execute()
        else:
            execute()
