# -*- coding: utf-8 -*-

import sandbox.common.types.misc as ctm
import sandbox.projects.common.binary_task as binary_task
import sandbox.projects.sandbox.resources as sb_resources
import sandbox.sdk2 as sdk2
from sandbox.common.types import resource
from sandbox.projects.qafw.common import qemulauncher, vnclauncher
from sandbox.projects.qafw.common.ggr_ui import GgrUiComponent
from sandbox.projects.qafw.common.gogridrouter import GoGridRouterBrowserItem
from sandbox.projects.qafw.common.gogridrouter import GoGridRouterConfig
from sandbox.projects.qafw.common.gogridrouter import \
    GoGridRouterConfigurableComponent
from sandbox.projects.qafw.common.gogridrouter import GoGridRouterGroupItem
from sandbox.projects.qafw.common.gogridrouter import GoGridRouterHostItem
from sandbox.projects.qafw.resource_types import SeleniumGgrUi
from sandbox.projects.qafw.resource_types import SeleniumGridRouter
from sandbox.projects.qafw.resource_types import SeleniumQemuImage


class WebDriverQemu(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Runs selenoid task for qemu vm.
    """

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        container = sdk2.parameters.Container(
            "Environment container resource",
            default_value="1293783910",
            resource_type=sb_resources.LXC_CONTAINER,
            platform="linux_ubuntu_16.04_xenial",
            required=True,
        )

        ggr = sdk2.parameters.LastReleasedResource(
            "GGR resource",
            description="Resource to use in task",
            resource_type=SeleniumGridRouter,
            required=False
        )

        ggr_ui = sdk2.parameters.LastReleasedResource(
            "GGR UI resource",
            description="Resource to build charts",
            resource_type=SeleniumGgrUi,
            required=False
        )

        with sdk2.parameters.Group('Project build') as project_build_block:
            build_id = sdk2.parameters.Integer(
                'Build ID',
                hint=True,
                default=0,
            )

        image = sdk2.parameters.LastResource(
            "QEMU image resource",
            description="Resource to use in task",
            resource_type=SeleniumQemuImage,
            required=False
        )

        count = sdk2.parameters.Integer(
            "Count of running qemu instances",
            hint=True,
            default=1,
        )

        enable_vnc = sdk2.parameters.Bool(
            "Start VNC for VM",
            description="Start VNC for VM",
            default_value=True,
        )

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Output():
            endpoint = sdk2.parameters.String(
                "WebDriver address",
            )
            status_endpoint = sdk2.parameters.String(
                "/status API address",
            )

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "QAFW_SANDBOX_BINARY", "released": self.Parameters.binary_executor_release_type},
            "state": [resource.State.READY]
        }

    def _ggr_bin(self):
        return str(sdk2.ResourceData(self.Parameters.ggr).path)

    def _ggr_ui_bin(self):
        return str(sdk2.ResourceData(self.Parameters.ggr_ui).path)

    def on_execute(self):
        self._go_grid_router = None
        self._ggr_ui = None
        self.qemu_list = []
        self.vnc_list = []
        browser_item = GoGridRouterBrowserItem(self.Parameters.image.browser, [self.Parameters.image.version])
        group_item = GoGridRouterGroupItem(browsers=[browser_item])
        config = GoGridRouterConfig(groups=[group_item])
        self._go_grid_router = GoGridRouterConfigurableComponent(self._ggr_bin(), cwd=self.path(), config=config)
        try:
            image_path = str(sdk2.ResourceData(self.Parameters.image).path)
            for index in xrange(int(self.Parameters.count)):
                qemu = qemulauncher.QemuLauncher(
                    image_path,
                    display=100 + index,
                    vm_cores=self.Parameters.image.ncpu,
                    vm_ram=self.Parameters.image.ram,
                    processes=self.Parameters.image.processes)
                self.qemu_list.append(qemu)
                qemu.start()
                for endpoint in qemu.endpoints:
                    group_item.endpoints.append(GoGridRouterHostItem("[{}]".format(str(endpoint[0])), endpoint[1], 1))

                if self.Parameters.enable_vnc:
                    vnc = vnclauncher.VncLauncher(index)
                    vnc.start()
                    vnc.wait()
                    self.vnc_list.append(vnc)

            for qemu in self.qemu_list:
                qemu.wait()

            self._go_grid_router.start()
            self._go_grid_router.wait()

            self._ggr_ui = GgrUiComponent(self._ggr_ui_bin(), cwd=self.path(), ggr=self._go_grid_router)
            self._ggr_ui.start()
            self._ggr_ui.wait()

            self.Parameters.status_endpoint = "{}:{}".format(self._ggr_ui.host, self._ggr_ui.port)
            self.Parameters.endpoint = "{}:{}".format(self._go_grid_router.host, self._go_grid_router.port)
            self.set_info("Webdriver endpoint = {}, status endpoint = {}".format(self.Parameters.endpoint, self.Parameters.status_endpoint))

            for qemu in self.qemu_list:
                qemu.wait_while_running()

            self._go_grid_router.wait_while_running()
        finally:
            self._stop()

    def _stop(self):
        for qemu in self.qemu_list:
            qemu.stop()
        for vnc in self.vnc_list:
            vnc.stop()
        if self._go_grid_router:
            self._go_grid_router.stop()

    def on_terminate(self):
        self.stop()

    def on_before_timeout(self, seconds):
        self.stop()
