# -*- coding: utf-8 -*-

import datetime

import StringIO
import paramiko
import sandbox.common.types.resource as ctr
import sandbox.sdk2.helpers
from sandbox import sdk2
from sandbox.projects.juggler.resource_types import JUGGLER_CLIENT_BINARY


class BasicMetricAggregator(sdk2.Resource):
    releasable = True
    any_arch = True
    auto_backup = True
    releasers = ['crastin', 'QLOUD-DEV']

    build_version = sdk2.parameters.String("Version")
    build_commit = sdk2.parameters.String("Commit")
    build_date = sdk2.parameters.String("Date")
    build_branch = sdk2.parameters.String("Branch")


class FlameMetricAggregator(BasicMetricAggregator):
    pass


class BuildMetricAggregator(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        source_branch = sdk2.parameters.String("git branch", default="master", required=True)
        agent_hostname = sdk2.parameters.String("agent hostname", required=True)
        juggler_client_binary = sdk2.parameters.LastReleasedResource(
            'juggler client to include',
            resource_type=JUGGLER_CLIENT_BINARY,
            state=ctr.State.READY,
            attrs=dict(released='stable'),
        )

    def on_execute(self):
        username = 'root'
        bitbucket_token = sdk2.Vault.data(self.owner, 'qloud_ci_bb_token')

        private_key_object = StringIO.StringIO(sdk2.Vault.data(self.owner, 'qloud_ci_ssh_private_key'))
        ssh_password = sdk2.Vault.data(self.owner, 'qloud_ci_ssh_password')
        private_key = paramiko.RSAKey(file_obj=private_key_object, password=ssh_password)

        with sandbox.sdk2.helpers.ProcessLog(self, logger="remote_build") as pl:
            pl.logger.propagate = 1
            build_remote(hostname=self.Parameters.agent_hostname, username=username, private_key=private_key,
                         ssh_password=ssh_password, bitbucket_token=bitbucket_token,
                         branch=self.Parameters.source_branch, logger=pl.logger)

        with open('version.txt') as version:
            build_commit = version.readline().strip()
            build_version = version.readline().strip()
        build_date = datetime.datetime.now().strftime('%Y-%m-%d_%H:%M:%S_UTC')
        branch_suffix = self.Parameters.source_branch.split('/')[-1]

        BasicMetricAggregator(
            self, 'metric-aggregator', 'aggregator', build_version=build_version, build_commit=build_commit,
            build_date=build_date, build_branch=branch_suffix
        )

        FlameMetricAggregator(
            self, 'metric-aggregator', 'flame.pl', build_version=build_version, build_commit=build_commit,
            build_date=build_date, build_branch=branch_suffix
        )

        self.set_info('Built version #{} commit {}'.format(build_version, build_commit))


def build_remote(hostname, username, private_key, ssh_password, bitbucket_token, branch, logger):
    with paramiko.SSHClient() as client:
        client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        logger.info("connecting to remote host {}@{} to build branch {}".format(username, hostname, branch))
        client.connect(hostname, username=username, pkey=private_key, password=ssh_password)

        tran = client.get_transport()
        chan = tran.open_session()
        chan.get_pty()
        with chan.makefile() as f:
            try:
                chan.exec_command('''
                    rm -rf metric-aggregator &&
                    git clone https://x-oauth-token:{}@bb.yandex-team.ru/scm/qloud/metric-aggregator.git --branch {} &&
                    cd metric-aggregator &&
                    git rev-parse --short HEAD > version.txt &&
                    git log --pretty=oneline | wc -l >> version.txt &&
                    make all
                '''.format(bitbucket_token, branch))
            except Exception as e:
                logger.info('Remote build failed {}'.format(str(e)))
                raise
            finally:
                logger.info(f.read())

            with client.open_sftp() as sftp:
                sftp.get('metric-aggregator/aggregator', 'aggregator')
                sftp.get('metric-aggregator/flame.pl', 'flame.pl')
                sftp.get('metric-aggregator/version.txt', 'version.txt')
