# coding: utf-8

import sandbox.common.types.misc as ctm
from sandbox.projects.common.nanny import nanny
from sandbox import sdk2
import os
import logging


class QloudNginxExecutable(sdk2.resource.Resource):
    """Nginx"""
    auto_backup = True
    any_arch = False
    releasable = True
    executable = True
    releasers = ['torkve', 'QLOUD']

    version = sdk2.Attributes.String("Bundle version", required=True, default="0.0")


class BuildQloudNginx(nanny.ReleaseToNannyTask2, sdk2.Task):
    '''Build mainline version of Nginx statically from sources'''

    resource_type = QloudNginxExecutable

    class Parameters(sdk2.Task.Parameters):
        description = 'Mainline version of nginx (static build)'

        nginx_source_version = sdk2.parameters.String('version of nginx',
                                                      required=True,
                                                      default='1.13.10')
        libressl_source_version = sdk2.parameters.String('version of LibreSSL',
                                                         required=True,
                                                         default='2.7.0')
        pcre_source_version = sdk2.parameters.String('version of PCRE',
                                                     required=True,
                                                     default='8.42')
        zlib_source_version = sdk2.parameters.String('version of zlib',
                                                     required=True,
                                                     default='1.2.11')

        with sdk2.parameters.Output:
            nginx_resource = sdk2.parameters.Resource('Final resource', required=True)

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    def on_execute(self):
        nginx_source_url = 'http://nginx.org/download/nginx-%s.tar.gz'
        nginx_source_ver = str(self.Parameters.nginx_source_version)

        libressl_source_url = 'http://ftp.openbsd.org/pub/OpenBSD/LibreSSL/libressl-%s.tar.gz'
        libressl_source_ver = str(self.Parameters.libressl_source_version)

        pcre_source_url = 'ftp://ftp.csx.cam.ac.uk/pub/software/programming/pcre/pcre-%s.tar.gz'
        pcre_source_ver = str(self.Parameters.pcre_source_version)

        zlib_source_url = 'http://zlib.net/zlib-%s.tar.gz'
        zlib_source_ver = str(self.Parameters.zlib_source_version)

        current_dir = os.path.abspath('./')
        libressl_dir = os.path.join(current_dir, 'libressl-%s' % libressl_source_ver)
        nginx_dir = os.path.join(current_dir, 'nginx-%s' % nginx_source_ver)
        pcre_dir = os.path.join(current_dir, 'pcre-%s' % pcre_source_ver)
        zlib_dir = os.path.join(current_dir, 'zlib-%s' % zlib_source_ver)

        for name, url, version, tarname in (
            ('libressl', libressl_source_url, libressl_source_ver, 'libressl-%s.tar.gz'),
            ('pcre', pcre_source_url, pcre_source_ver, 'pcre-%s.tar.gz'),
            ('zlib', zlib_source_url, zlib_source_ver, 'zlib-%s.tar.gz'),
            ('nginx', nginx_source_url, nginx_source_ver, 'nginx-%s.tar.gz'),
        ):
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('get_%s' % name)) as pl:
                proc = sdk2.helpers.subprocess.Popen(
                    ['wget', '-q', url % version],
                    stdout=pl.stdout,
                    stderr=sdk2.helpers.subprocess.STDOUT,
                )
                proc.wait()
                assert proc.returncode == 0
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('tar_%s' % name)) as pl:
                proc = sdk2.helpers.subprocess.Popen(
                    ['tar', '-xf', tarname % version],
                    stdout=pl.stdout,
                    stderr=sdk2.helpers.subprocess.STDOUT,
                )
                proc.wait()
                assert proc.returncode == 0

        os.chdir(libressl_dir)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('configure_libressl')) as pl:
            proc = sdk2.helpers.subprocess.Popen(
                ['./configure', 'LDFLAGS=-lrt', '--prefix=%s/.openssl/' % libressl_dir],
                stdout=pl.stdout,
                stderr=sdk2.helpers.subprocess.STDOUT,
                cwd=libressl_dir,
            )
            proc.wait()
            assert proc.returncode == 0
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('make_libressl')) as pl:
            proc = sdk2.helpers.subprocess.Popen(
                ['make', 'install-strip'],
                stdout=pl.stdout,
                stderr=sdk2.helpers.subprocess.STDOUT,
                cwd=libressl_dir,
            )
            proc.wait()
            assert proc.returncode == 0

        os.chdir(zlib_dir)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('configure_zlib')) as pl:
            proc = sdk2.helpers.subprocess.Popen(
                ['./configure', '--prefix=%s/.zlib' % zlib_dir],
                stdout=pl.stdout,
                stderr=sdk2.helpers.subprocess.STDOUT,
                cwd=zlib_dir,
            )
            proc.wait()
            assert proc.returncode == 0
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('make_libressl')) as pl:
            proc = sdk2.helpers.subprocess.Popen(
                ['make', 'install'],
                stdout=pl.stdout,
                stderr=sdk2.helpers.subprocess.STDOUT,
                cwd=zlib_dir,
            )
            proc.wait()
            assert proc.returncode == 0

        os.chdir(nginx_dir)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('configure_nginx')) as pl:
            proc = sdk2.helpers.subprocess.Popen(
                [
                    './configure',
                    '--with-openssl=%s' % libressl_dir,
                    '--conf-path=nginx-conf/nginx.conf',
                    '--http-client-body-temp-path=./tmp/body',
                    '--http-fastcgi-temp-path=./tmp/fastcgi',
                    '--http-proxy-temp-path=./tmp/proxy',
                    '--http-scgi-temp-path=./tmp/scgi',
                    '--http-uwsgi-temp-path=./tmp/uwsgi',
                    '--lock-path=./tmp/nginx.lock',
                    '--pid-path=./tmp/nginx.pid',
                    '--with-cc-opt=-O2',
                    '--with-file-aio',
                    '--with-http_gunzip_module',
                    '--with-http_gzip_static_module',
                    '--with-http_realip_module',
                    '--with-http_slice_module',
                    '--with-http_ssl_module',
                    '--with-http_stub_status_module',
                    '--with-http_v2_module',
                    '--with-ld-opt=-Wl,--no-as-needed,-lrt',
                    '--with-pcre-jit',
                    '--with-pcre=%s' % pcre_dir,
                    '--with-stream',
                    '--with-stream_realip_module',
                    '--with-stream_ssl_module',
                    '--with-stream_ssl_preread_module',
                    '--with-threads',
                    '--with-zlib=%s' % zlib_dir,
                    '--without-http_ssi_module',
                    '--without-mail_imap_module',
                    '--without-mail_pop3_module',
                    '--without-mail_smtp_module',
                    '--without-poll_module',
                    '--without-select_module'
                ],
                stdout=pl.stdout,
                stderr=sdk2.helpers.subprocess.STDOUT,
                cwd=nginx_dir,
            )
            proc.wait()
            assert proc.returncode == 0

            proc = sdk2.helpers.subprocess.Popen(
                ['touch', '%s/.openssl/include/openssl/ssl.h' % libressl_dir],
                stdout=pl.stdout,
                stderr=sdk2.helpers.subprocess.STDOUT,
                cwd=libressl_dir,
            )
            proc.wait()
            assert proc.returncode == 0

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('make_nginx')) as pl:
            proc = sdk2.helpers.subprocess.Popen(
                ['make', '-j1'],
                stdout=pl.stdout,
                stderr=sdk2.helpers.subprocess.STDOUT,
                cwd=nginx_dir,
            )
            proc.wait()
            assert proc.returncode == 0

        self.Parameters.nginx_resource = self.resource_type(
            self,
            "nginx %s" % (self.Parameters.nginx_source_version,),
            "%s/objs/nginx" % nginx_dir,
            version=self.Parameters.nginx_source_version,
            arch='linux',
        )
