# coding: utf-8

import os
import logging
import textwrap

from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.common.nanny import nanny
from sandbox.projects.porto.common.resource_types import PORTO_LAYER_SEARCH_UBUNTU_XENIAL_APP
from sandbox.projects.resource_types import OTHER_RESOURCE
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import sandbox.common.types.client as ctc


class QnotifierPortoLayer(sdk2.resource.Resource):
    """qnotifier layer"""
    auto_backup = True
    any_arch = False
    releasable = True
    executable = True
    releasers = ['torkve', 'QLOUD']

    version = sdk2.Attributes.String("Bundle version", required=True, default="0.0")


class BuildQnotifierLayer(nanny.ReleaseToNannyTask2, sdk2.Task):
    """Build qnotifier from sources"""

    class Parameters(sdk2.Task.Parameters):
        description = 'qnotifier build'

        version = sdk2.parameters.String(
            "version of qnotifier",
            required=True,
            default='0.0'
        )

        svn_url = sdk2.parameters.ArcadiaUrl(
            "qnotifier sources url",
            required=True,
            default_value=Arcadia.ARCADIA_TRUNK_URL + "/infra/qnotifier"
        )

        parent_layer = sdk2.parameters.LastReleasedResource(
            "Parent layer name",
            resource_type=PORTO_LAYER_SEARCH_UBUNTU_XENIAL_APP,
            state=(ctr.State.READY,),
            required=True,
        )

        with sdk2.parameters.Output:
            qnotifier_resource = sdk2.parameters.Resource('qnotifier layer', required=True)

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.PORTOD
        privileged = True

    def on_execute(self):
        qnotifier_data_resource = OTHER_RESOURCE(
            self, self.Parameters.description, 'qnotifier.tar.gz'
        )
        qnotifier_data = sdk2.ResourceData(qnotifier_data_resource)

        Arcadia.export(self.Parameters.svn_url, "qnotifier")
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('tar')) as pl:
            proc = sdk2.helpers.subprocess.Popen(
                ['tar', 'czf', os.path.abspath(str(qnotifier_data.path)), 'qnotifier'],
                stdout=pl.stdout,
                stderr=sdk2.helpers.subprocess.STDOUT,
            )
            proc.wait()
            assert proc.returncode == 0

        qnotifier_data.ready()

        parent_layer = sdk2.ResourceData(self.Parameters.parent_layer)
        parent_layer_path = parent_layer.path

        with open('script.sh', 'w') as f:
            f.write(textwrap.dedent(
                '''
                #!/bin/sh

                set -e
                set -x

                apt-get update
                apt-get --yes --purge autoremove
                apt-get install --yes --no-install-recommends python3.5 python3-virtualenv python3-venv libsqlite3-dev gcc-5 gcc python3-dev

                curl https://proxy.sandbox.yandex-team.ru/%s | tar xvz
                cd qnotifier
                python3.5 -m venv snapshot
                source snapshot/bin/activate
                pip install -i https://pypi.yandex-team.ru/simple -r requirements.txt
                pip install -i https://pypi.yandex-team.ru/simple -r requirements-worker.txt
                chown -R loadbase ../qnotifier

                apt-get --yes --purge remove gcc-5 gcc python3-dev
                apt-get --yes --purge autoremove
                rm -fv /var/cache/apt/archives/*.deb
                rm -rfv /var/lib/apt/lists/*
                rm -fv /var/cache/apt/*.bin
                ''' % (qnotifier_data_resource.id,)
            ))

        args = ['portoctl', 'build',
                '-S', os.path.abspath('script.sh'),
                '-L', str(parent_layer_path),
                '-o', 'PORTO_LAYER_QNOTIFIER.tar.xz',
                'bind_dns=false',
                'net=inherited',
                'aging_time=3600',
                'enable_porto=false',
                'hostname=sandbox-' + str(self.id),
                'virt_mode=app',
                'private=sandbox-' + str(self.id),
                'space_limit=8192M',
                'memory_limit=4096M',
                ]

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('portoctl')) as pl:
            proc = sdk2.helpers.subprocess.Popen(
                args,
                stdout=pl.stdout,
                stderr=sdk2.helpers.subprocess.STDOUT,
            )
            proc.wait()
            assert proc.returncode == 0

        self.Parameters.qnotifier_resource = QnotifierPortoLayer(
            self,
            "qnotifier %s" % self.Parameters.version,
            "PORTO_LAYER_QNOTIFIER.tar.xz",
            version=self.Parameters.version,
            arch='linux',
        )
