import requests
import logging
import shutil
import time

from sandbox import sdk2
from sandbox.common.types.misc import DnsType

MAX_RETIRES = 8
FIRST_RETRY_SECONDS = 2


class CopyDockerLayer(sdk2.Task):
    """
    Downloads a docker layer via http

    Parameters:
        * http - http link for downloading layer
        * digest - hash of that layer (see https://docs.docker.com/registry/spec/api/ format description)
    """

    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        http = sdk2.parameters.String("http")
        http.required = True

        digest = sdk2.parameters.String("digest")
        digest.required = True

    def download_and_post_layer(self, http_url, filename):
        logging.info("downloading {} by {}".format(filename, http_url))
        with requests.get(http_url, stream=True) as response:
            response.raise_for_status()

            resource = sdk2.Resource["PORTO_LAYER"](self, 'Porto Layer for Docker Image', filename)
            resource_data = sdk2.ResourceData(resource)
            with resource_data.path.open('wb') as f:
                shutil.copyfileobj(response.raw, f)
            resource_data.ready()

    def on_execute(self):
        http = str(self.Parameters.http)
        filename = str(self.Parameters.digest).replace(':', '_') + ".gz"

        for i in range(MAX_RETIRES + 1):
            try:
                self.download_and_post_layer(http, filename)
                break
            except requests.exceptions.HTTPError as e:
                logging.error("http download failed: " + str(e))
                if i == MAX_RETIRES:
                    raise e
                time.sleep(FIRST_RETRY_SECONDS * (2 ** i))
