import logging
import os


from sandbox import sdk2
from sandbox.projects.quality.resources import resources as quality_resources
from sandbox.projects.common import context_managers
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.constants import constants as sdk_constants
from sandbox.projects.common import binary_task
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.vcs.svn import Arcadia


FACTORS_DEPENDENCIES_DUMP_FILE_NAME = "factors_dependencies_dump.proto.txt"

DEPENDENCY_GRAPH_DIR = "quality/relev_tools/dependency_graph"
DEPENDENCY_GRAPH_BINARY_PATH = os.path.join(DEPENDENCY_GRAPH_DIR, "dependency_graph")


class DumpFactorsDependencies(binary_task.LastBinaryTaskRelease, sdk2.Task):
    __logger = logging.getLogger("TASK_LOGGER")
    __logger.setLevel(logging.DEBUG)

    class Parameters(sdk2.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia url",
            default_value=Arcadia.ARCADIA_TRUNK_URL,
        )
        tasks_archive_resource = binary_task.binary_release_parameters(stable=True)

    def get_arcadia_src_dir(self):
        if arcadia_sdk.fuse_available():
            return arcadia_sdk.mount_arc_path(self.Parameters.arcadia_url, fallback=True)
        return context_managers.nullcontext(Arcadia.get_arcadia_src_dir(self.Parameters.arcadia_url))

    def on_execute(self):
        factors_dependencies_dump = quality_resources.FactorsDependenciesDump(
            self,
            "Factor dependencies dump",
            FACTORS_DEPENDENCIES_DUMP_FILE_NAME,
            ttl="inf"
        )

        self.__logger.info("Preparing arcadia directory: build dependency_graph binary")
        with self.get_arcadia_src_dir() as arcadia_dir:
            arcadia_sdk.do_build(
                build_system=sdk_constants.YMAKE_BUILD_SYSTEM,
                source_root=arcadia_dir,
                targets=[DEPENDENCY_GRAPH_DIR],
                results_dir=arcadia_dir,
                clear_build=False,
            )

            self.__logger.info("Dumping factors dependencies")
            factor_dependencies_controller_binary = os.path.join(arcadia_dir, DEPENDENCY_GRAPH_BINARY_PATH)
            command = [
                factor_dependencies_controller_binary,
                "dump_mappings_used_factors",
            ]
            with sdk2.helpers.ProcessLog(self, logger="factors_dependencies_dumper") as factors_dependencies_dumper_pl:
                with open(FACTORS_DEPENDENCIES_DUMP_FILE_NAME, "w") as factors_dependencies_dump_file:
                    subprocess.check_call(command, stdout=factors_dependencies_dump_file,
                                          stderr=factors_dependencies_dumper_pl.stderr)

        sdk2.ResourceData(factors_dependencies_dump).ready()
