import datetime
import json
import logging
import os
import subprocess

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.common import binary_task
from sandbox.projects.quality.resources import resources as rs
import sandbox.projects.quality.tasks.ValidateFactorsConfig as vfc
from sandbox.common.errors import TaskFailure

from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.constants import constants as sdk_constants


FACTORS_CONFIG_FILE_NAME = "conf.json"


class GetFactorsConfig(sdk2.Task):
    """ Checkout new factors config. """

    class Parameters(sdk2.Parameters):
        kill_timeout = 3600

        arc_token_owner = sdk2.parameters.String("Group owner of arc token", default="WEB_RELEVANCE", required=True, multiline=False)

        yt_token_vault_owner = sdk2.parameters.String('Vault owner for yt token')
        yt_token_vault_name = sdk2.parameters.String('Vault name for yt token')
        publish_to_yt = sdk2.parameters.Bool('Publish to YT', default=True)
        run_validation_task = sdk2.parameters.Bool('Run config validation taks', default=True)

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia url",
            default_value='arcadia-arc:/#trunk',
        )

    def _build(self, target, output_directory):
        token = sdk2.Vault.data(self.Parameters.arc_token_owner, name='ARC_TOKEN')
        with arcadiasdk.mount_arc_path(self.Parameters.arcadia_url, use_arc_instead_of_aapi=True, arc_oauth_token=token) as arc_root:
            arcadiasdk.do_build(
                build_system=sdk_constants.SEMI_DISTBUILD_BUILD_SYSTEM,
                source_root=arc_root,
                targets=[target],
                results_dir=output_directory,
                clear_build=False,
            )

    def on_execute(self):
        import yt.wrapper as yt

        factors_config = rs.FactorsConfig(
            self,
            'FactorsConfig',
            FACTORS_CONFIG_FILE_NAME,
        )

        with self.memoize_stage.try_to_get_config:
            target = 'quality/relev_tools/resolve_factors/create_json_config/bin'
            binary_name = 'slices_config'
            output_directory = 'conf'
            logging.info("start of building...")
            self._build(target, output_directory)
            logging.info("building completed.")
            logging.info(["output_directory: %s", os.listdir(output_directory)])

            with sdk2.helpers.ProcessLog(self, logger="factors_config_dumper") as factors_config_dumper_pl:
                with open(FACTORS_CONFIG_FILE_NAME, "w") as factors_config_file:
                    cmd = os.path.join(os.curdir, output_directory, target, binary_name)
                    p = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=factors_config_dumper_pl.stderr)
                    factors_slices = json.load(p.stdout)
                    json.dump(factors_slices, factors_config_file, indent=4, sort_keys=True)
            sdk2.ResourceData(factors_config).ready()

        with self.memoize_stage.validate_config:
            if self.Parameters.run_validation_task:
                subtasks = vfc.ValidateFactorsConfig(
                    None,
                    description="Test new config",
                    owner=self.owner,
                    tested_config=factors_config.id,
                ).enqueue()
                self.Context.subtask = subtasks.id
                raise sdk2.WaitTask(subtasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, timeout=60 * 60 * 3)

        with self.memoize_stage.approve_config:
            if self.Parameters.run_validation_task:
                logging.info("Check task %s" % self.Context.subtask)
                subtask = self.server.task[self.Context.subtask].read()
                if subtask["status"] != 'SUCCESS':
                    logging.error("Subtask {} finished with status {}".format(subtask["id"], subtask["status"]))
                    raise TaskFailure("Validating subtask has failed")

            prepared_config = str(sdk2.ResourceData(sdk2.Resource[factors_config.id]).path)
            time_utc = datetime.datetime.utcnow().isoformat()

            if self.Parameters.publish_to_yt:
                with open(prepared_config, 'r') as config_file:
                    token = sdk2.Vault.data(self.Parameters.yt_token_vault_owner, self.Parameters.yt_token_vault_name)
                    factors_config_content = config_file.read()
                    yt_configs_dir = '//home/factordev/production/resolve_factors_configs/'
                    for cluster_name in ['hahn', 'arnold']:
                        yt_client = yt.YtClient(cluster_name, token)
                        with yt.Transaction(client=yt_client, timeout=60000):
                            yt_client.move(yt_configs_dir + 'current', yt_configs_dir + 'prev', force=True)
                            yt_client.write_file(yt_configs_dir + 'current', factors_config_content)
                            yt_client.set(yt_configs_dir + 'current/@sandbox_task_id', self.id)
                            yt_client.set(yt_configs_dir + 'current/@utc_time', time_utc)

            logging.info("Success testing!")
            factors_config.valid = True
            factors_config.utc_time = time_utc
