from sandbox.common.errors import TaskFailure
from sandbox import sdk2
from sandbox.projects.quality.resources import resources as rs
import json
import logging


class ValidateFactorsConfig(sdk2.Task):
    """
    Validate new factors config.
    Logs jsondiff output for each slice in configs.
    """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 60 * 60 * 3

        tested_config = sdk2.parameters.Resource(
            'Factors tested config',
            resource_type=rs.FactorsConfig,
            required=True
        )
        released_config = sdk2.parameters.Resource(
            'Factors last released config',
            resource_type=rs.FactorsConfig,
            required=False
        )

    def _test_config(self, released_path, tested_path):
        from jsondiff import diff

        with open(released_path, 'r') as fr, open(tested_path, 'r') as ft:
            released = json.load(fr)
            tested = json.load(ft)

        slice_names = set(tested.keys()).union(set(released.keys()))
        fail_on_slice = False

        for slice_name in slice_names:
            logging.info(slice_name)
            released_slice = released.get(slice_name)
            tested_slice = tested.get(slice_name)

            approximate_diffs = sum(released_slice[i] != tested_slice[i]
                                    for i in range(min(len(tested_slice), len(released_slice))))

            # jsondiff is too slow
            if approximate_diffs > 50:
                fail_on_slice = True
                logging.info('Approximate diffs count is greater than 50, can not produce jsondiff result')
                continue

            try:
                slices_diff = json.loads(diff(released_slice, tested_slice, syntax='symmetric', dump=True))
            except Exception as e:
                logging.info(e)
                logging.info('jsondiff has failed, approximate diff: %s' % approximate_diffs)
                if approximate_diffs > 20:
                    fail_on_slice = True
                continue

            total_factors = len(released_slice) + len(tested_slice)
            diffed_factors = 2 * len(slices_diff)

            fail_on_slice = fail_on_slice or diffed_factors > 20 or diffed_factors > total_factors * 0.2

            logging.info('Total factors: %s, diffed factors: %s' % (total_factors / 2, diffed_factors / 2))

            logging.info(json.dumps(slices_diff, indent=4, sort_keys=True))

        return fail_on_slice

    def on_execute(self):
        last_released_config = self.Parameters.released_config
        if last_released_config is None:
            last_released_config = sdk2.Resource.find(
                type=rs.FactorsConfig,
                state="READY",
                attrs={"valid": True}
            ).first()
            if last_released_config is None:
                return
        logging.info("Last released resource id: {}".format(last_released_config.id))
        last_released_config = str(sdk2.ResourceData(last_released_config).path)
        tested_config = str(sdk2.ResourceData(self.Parameters.tested_config).path)
        if self._test_config(last_released_config, tested_config):
            raise TaskFailure("Factors config is broken")
