import os
import shutil
import tempfile
import sys

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import process
from sandbox.sdk2 import yav


class SystemPythonVirtualEnvironment(object):
    def __init__(self, requirements):
        self._dir = tempfile.mkdtemp()
        process.run_process(
            cmd=[
                environments.VirtualEnvironment.VIRTUALENV_EXE,
                '--python', '/usr/bin/python2',
                self._dir
            ],
            log_prefix='venv creation',
            check=True,
        )
        process.run_process(
            cmd=[
                os.path.join(self._dir, 'bin', 'pip'),
                'install',
                '--index', environments.VirtualEnvironment.PYPI_URLS[0],
                '--upgrade',
                'pip<21', 'setuptools' if sys.version_info[0] > 2 else 'setuptools<45',  # python 2 no longer works with setuptools>=45
            ],
            log_prefix='venv update',
            check=True,
        )
        process.run_process(
            cmd=[
                os.path.join(self._dir, 'bin', 'pip'),
                'install',
                '--index', environments.VirtualEnvironment.PYPI_URLS[0],
                '--upgrade',
                '-vvv',
            ] + requirements.split(),
            log_prefix='venv requirements',
            check=True,
        )
        self.executable = os.path.join(self._dir, 'bin', 'python')

    def __enter__(self):
        return self

    def __exit__(self, *exc):
        shutil.rmtree(self._dir)


class QuasarBugReportFastTickets(sdk2.Task):
    class Parameters(sdk2.Parameters):

        general_options = sdk2.parameters.Info('General options')
        dry_run = sdk2.parameters.Bool('Do not make startrack tickets', default=False)
        robot_secrets_version = sdk2.parameters.String('Yav robot-quasar secret version', default='ver-01e3vkfa3g3zx1beh5skbj9sex')

        yt_options = sdk2.parameters.Info('YT options')
        yt_cluster = sdk2.parameters.String('YT cluster', default='hahn')
        yt_token_vault = sdk2.parameters.String('YT token vault', default='robot-quasar-yt-token')
        yt_table_regexp = sdk2.parameters.String('YT table regexp', default='//home/quasar-dev/bug_reports_1h/.+')

        status_string_path = sdk2.parameters.String('Status string', default='//home/quasar-dev/bug_report_status_1h')

        startrek_options = sdk2.parameters.Info('Startrek options')
        startrek_token_vault = sdk2.parameters.String('Token vault', default='robot-quasar-startrek-token')
        startrek_queue = sdk2.parameters.String('Queue', default='QBRTEST')
        startrek_followers = sdk2.parameters.List('Followers', default=[])
        startrek_author_as_follower = sdk2.parameters.Bool('Add bugreport author as follower', default=False)

        setrace_options = sdk2.parameters.Info('Setrace options')
        setrace_enabled = sdk2.parameters.Bool('Put recent setraces in ticket', default=False)
        setrace_url = sdk2.parameters.String('Setrace url', default='https://setrace.yandex-team.ru')

        tvm2_options = sdk2.parameters.Info('TVM2 options')
        tvm2_client_id = sdk2.parameters.String('Client id', default='2012288')
        tvm2_secret_vault = sdk2.parameters.String('Secret vault', default='quasar-sandbox-blackbox-tvm')

    class Requirements(sdk2.Requirements):
        class Caches(sdk2.Requirements.Caches):
            pass

        cores = 1

    def on_execute(self):
        robot_secrets = yav.Secret("sec-01d2ffwrdbwyj37zkj4r8zegsn", self.Parameters.robot_secrets_version)
        yt_cluster = self.Parameters.yt_cluster
        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_vault)
        yt_table_regexp = self.Parameters.yt_table_regexp
        status_string_path = self.Parameters.status_string_path
        startrek_token = sdk2.Vault.data(self.owner, self.Parameters.startrek_token_vault)
        startrek_queue = self.Parameters.startrek_queue
        startrek_followers = self.Parameters.startrek_followers
        startrek_author_as_follower = self.Parameters.startrek_author_as_follower
        tvm2_client_id = self.Parameters.tvm2_client_id
        tvm2_secret = sdk2.Vault.data(self.owner, self.Parameters.tvm2_secret_vault)
        setrace_url = self.Parameters.setrace_url
        setrace_token = robot_secrets.data()["robot-quasar-setrace-token"]
        setrace_enabled = self.Parameters.setrace_enabled

        with SystemPythonVirtualEnvironment('requests[security]==2.25.1 startrek_client tvm2 yandex-yt pytz ticket_parser2') as venv:

            process.run_process(
                cmd=[
                    venv.executable,
                    os.path.join(os.path.dirname(__file__), 'tickets.py'),
                    '--yt_cluster', yt_cluster,
                    '--yt_table_regexp', yt_table_regexp,
                    '--status_string_path', status_string_path,
                    '--startrek_queue', startrek_queue,
                    '--startrek_followers', ','.join(startrek_followers),
                    '--tvm2_client_id', tvm2_client_id,
                    '--setrace_url', setrace_url,
                ] + (['--startrek_author_as_follower'] if startrek_author_as_follower else [])
                + (['--setrace_enabled'] if setrace_enabled else []),
                log_prefix='tickets',
                wait=True,
                check=True,
                environment={
                    'YT_TOKEN': yt_token,
                    'STARTREK_TOKEN': startrek_token,
                    'TVM2_SECRET': tvm2_secret,
                    'SETRACE_TOKEN': setrace_token
                },
            )
