import requests
import time
import logging
from frozendict import frozendict
from collections import Iterable

try:
    from urllib.parse import urlparse, urljoin, urlunparse, urlencode
except:
    from urlparse import urlparse, urljoin, urlunparse
    from urllib import urlencode
try:
    unicode_sc = unicode
except:
    unicode_sc = str


class SetraceClient(object):
    def __init__(self, setrace_token, setrace_base_url):
        self.setrace_token = setrace_token
        self.setrace_base_url = setrace_base_url

    def get_alice_trace(self, device_id, timestamp_from, timestamp_to):
        url = self._get_alice_url(device_id)
        response = requests.get(url,
                                headers={"Authorization": "OAuth {}".format(self.setrace_token)},
                                verify=False)

        # api has strange 201 return code. For further safity add 200 code as a successful
        if response.status_code not in (201, 200):
            logging.error(u"unable to get trace. Code was {}: {} : {}".format(response.status_code, url, response.content))
            return []
        result = []
        response_json = response.json()
        if "uuids" not in response_json:
            logging.error(u"no uuids field in response for {} : {}".format(url, response.content))
            return []

        for item in response.json()["uuids"]:
            trace_item = SetraceAliceSessionItem(**item)
            result.append(trace_item)

        result = [item for item in result if timestamp_from * 10 ** 6 <= item.req_ts < timestamp_to * 10 ** 6]

        for item in result:
            self.start_tracing(item)

        for item in result:
            self.wait_trace_result(item)
            item.last_recognized_phrase = self.fetch_last_pharse(item)

        return [r for r in sorted(result, key=lambda tr: -tr.req_ts)]

    def get_item_tree(self, search_id):
        url = self._get_item_tree_url(search_id)
        response = requests.get(url,
                                headers={"Authorization": "OAuth {}".format(self.setrace_token)},
                                verify=False)

        # api has strange 201 return code. For further safity add 200 code as a successful
        if response.status_code not in (200, 201):
            logging.error(u"unable to get trace. Code was {}: {} : {}".format(response.status_code, url, response.content))
            return None

        response_json = response.json()

        if not isinstance(response_json, dict):
            return None

        def get_children(json):
            if not isinstance(json, Iterable):
                return []
            children = []
            for item in json:
                item_children = []
                if "children" in item and len(item["children"]) > 0:
                    item_children = get_children(item["children"])
                kv = dict(**item)
                kv.update(children=item_children)

                children.append(SetraceSessionTreeItem(**kv))

            return children

        kv = dict(**response_json)
        kv.update(children=get_children(kv.get("children")))
        return SetraceSessionTreeItem(**kv)

    def wait_trace_result(self, item):
        response_json = {}
        for i in range(15):
            response = requests.get(self._get_search_status_url(item.req_id),
                                    headers={"Authorization": "OAuth {}".format(self.setrace_token)},
                                    verify=False)
            if response.status_code != 200:
                logging.error(u"Got error {} getting result for search {} : {}".format(response.status_code, item.req_id, response.content))

            response_json = response.json()
            if "result" not in response_json or response_json["result"] is None:
                logging.debug("No result...Continue {}".format(response_json))
                time.sleep(1)
                continue

            break

        if "result" not in response_json or response_json["result"] is None:
            logging.error("still got no trace result for {}".format(item.req_id))

    def start_tracing(self, item):
        """
         prefetch setrace makes its ttl = inf
         :return:
         """
        response = requests.get(item.url,
                                headers={"Authorization": "OAuth {}".format(self.setrace_token)},
                                verify=False)
        if response.status_code != 200:
            logging.error(u"unable to get trace {} : {}".format(item.url, response.content))

    def fetch_last_pharse(self, item):
        tree = self.get_item_tree(item.req_id)

        def tree_backward_it(tree):
            if len(tree.children) == 0:
                yield tree
                return

            stack = list()
            it = iter(tree.children[::-1])
            cur = next(it)
            while cur is not None:
                if len(cur.children) > 0:
                    stack.append((it, cur))
                    it = iter(cur.children[::-1])
                    cur = next(it)
                    continue

                yield cur

                next_cur = cur
                while next_cur == cur:
                    try:
                        next_cur = next(it)
                    except StopIteration:
                        if not len(stack):
                            return
                        it, parent = stack.pop()
                        yield parent
                        continue

                cur = next_cur

        for it in tree_backward_it(tree):
            if it.request_type == 'alice' and (it.search_type == "http_adapter-ALICE" or it.search_type == "unknown") and unicode_sc.startswith(it.title, u"vins:"):
                return it.title

        return None

    def _get_alice_url(self, device_id):
        url_parts = list(urlparse(urljoin(self.setrace_base_url, "v1/alice")))
        url_parts[4] = urlencode({"trace_by": device_id})

        return urlunparse(url_parts)

    def _get_item_tree_url(self, search_id):
        return urljoin(self.setrace_base_url, "v1/search/{}/results/tree".format(search_id))

    def _get_search_status_url(self, uuid):
        return urljoin(self.setrace_base_url, "v1/search/{}/status".format(uuid))

    def _get_start_trace_url(self, search_id):
        return urljoin(self.setrace_base_url, "ui/alice/tracePage/{}".format(search_id))


class SetraceAliceSessionItem(frozendict):
    def __init__(self, events_index_table, req_ts, reqid, setrace_time, url):
        super(SetraceAliceSessionItem, self).__init__(setrace_time=setrace_time,
                                                      req_id=reqid,
                                                      req_ts=req_ts,
                                                      events_index_table=events_index_table,
                                                      url=url)
        self.req_ts = req_ts
        self.setrace_time = setrace_time
        self.req_id = reqid
        self.url = url
        self.last_recognized_phrase = None


class SetraceSessionTreeItem(frozendict):
    def __init__(self, searchType, title="", children=[], requestType="root", **other):
        super(SetraceSessionTreeItem, self).__init__(search_type=searchType,
                                                     request_type=requestType,
                                                     title=title,
                                                     children=children,
                                                     **other)
        self.title = title
        self.search_type = searchType
        self.request_type = requestType
        self.children = children
