# coding=utf-8
import argparse
import collections
import datetime
import functools
import hashlib
import itertools
import json
import logging
import os
import re
import tempfile
import textwrap
import urllib


@functools.total_ordering
class Device(object):
    UNILOGVIEWER_URL = 'https://unilogviewer.voicetech.yandex-team.ru/?'
    UNILOGVIEWER_PRE_URL = 'https://unilogviewer-pre.common-ext.yandex-team.ru/?'
    UNILOGVIEWER_RTC_URL = 'https://unilogviewer.voicetech.yandex.net/?'

    def __init__(self, environment, device_id, device_type):
        self.environment = environment
        self.device_id = device_id
        self.device_type = device_type

    @property
    def id(self):
        return self.device_id

    @property
    def type(self):
        dt = self.device_type
        return dt[len('yandex'):] if dt.startswith('yandex') else dt

    @property
    def unilogviewer_url(self):
        uniproxy_url = self.environment.get('uniProxyUrl', 'alice')  # Default to prod in case of missing url
        if 'prestable' in uniproxy_url:
            return self.UNILOGVIEWER_PRE_URL
        if 'alice' in uniproxy_url:
            return self.UNILOGVIEWER_RTC_URL
        return self.UNILOGVIEWER_URL

    def __eq__(self, other):
        return self.id == other.id

    def __lt__(self, other):
        return self.id < other.id

    def __str__(self):
        env = self.environment.copy()
        envs = '\n'.join('    {}: {}'.format(k, v) for k, v in sorted(env.iteritems()))
        return '{device_type} / {device_id}\n{envs}'.format(device_type=self.device_type, device_id=self.device_id, envs=envs)


class BugReport(object):
    UNILOG_TS_DIFF = (-180, 120)

    def __init__(self):
        self.id = None
        self._account_id = None
        self.login = None
        self._timestamp = None
        self._devices = {}
        self._uuid = None
        self._text = None
        self._files = {}
        self._traces = {}
        self._deivce_id = None
        self._device_type = None

    @property
    def account_id(self):
        return self._account_id

    @account_id.setter
    def account_id(self, value):
        if self._account_id is not None and self._account_id != value:
            raise ValueError('Different account_ids for bug report: {} / {}'.format(self._account_id, value))
        self._account_id = value

    def add_device(self, value):
        self._devices[value.id] = value

    @property
    def devices(self):
        return self._devices.values()

    @property
    def text(self):
        return self._text

    @text.setter
    def text(self, value):
        if self._text is None or (value is not None and len(value) > len(self._text)):
            self._text = value

    @property
    def uuid(self):
        return self._uuid

    @uuid.setter
    def uuid(self, value):
        if self._uuid is not None and self._uuid != value:
            raise ValueError('Different uuids for bug report: {} / {}'.format(self._uuid, value))
        self._uuid = value

    @property
    def timestamp(self):
        return self._timestamp

    @timestamp.setter
    def timestamp(self, value):
        if self._timestamp is not None and self._timestamp <= value:
            return
        self._timestamp = value

    def add_file(self, name, value):
        full_name = name
        suffix = 1
        while full_name in self._files:
            full_name += '.' + str(suffix)
            suffix += 1
        self._files[full_name] = value

    def make_description(self):
        import pytz

        def format_timestamp(time_diff):
            local_tz = pytz.timezone('Europe/Moscow')
            return datetime.datetime.fromtimestamp(self.timestamp + time_diff, tz=local_tz)\
                .strftime('%d.%m.%y %H:%M')

        login = (u' от ' + self.login.make_signature()) if self.login else ''

        unilogviewer_params = urllib.urlencode({
            'timestamp': format_timestamp(self.UNILOG_TS_DIFF[0]) + ' - ' + format_timestamp(self.UNILOG_TS_DIFF[1]),
            'timezone': 'Europe/Moscow',
            'user_id': self.uuid,
        })

        if len(self._devices) == 1:
            device = self._devices[self._devices.keys()[0]]
            devices = str(device)
            unilogviewer = device.unilogviewer_url + unilogviewer_params
        else:
            devices = sorted(self._devices.itervalues())
            unilogviewer = devices[0].unilogviewer_url + unilogviewer_params
            devices = '\n  '.join(str(d) for d in devices)

        traces_txt_by_device = {platform_and_id: u"\n".join([u"{}{}".format(t.last_recognized_phrase + u": " if t.last_recognized_phrase is not None else "", "(({} {}))".format(t.url, t.setrace_time))
                                                             for t in device_traces])
                                for platform_and_id, device_traces in self._traces.items()}

        traces_txt = '\n\n'.join([u"{}:\n {}".format(k, v) for k, v in traces_txt_by_device.items()])

        return textwrap.dedent(u"""\
        Баг репорт{login} в {ts}:
        <[{text}]>

        (({unilogviewer} Посмотреть в UniLogViewer))

        {setraces}

        <{{Служебные данные
        id: {id}
        devices:
          {devices}
        uuid: {uuid}
        account_id: {account_id}
        }}>""").format(
            login=login,
            account_id=self.account_id,
            ts=datetime.datetime.fromtimestamp(self.timestamp),
            unilogviewer=unilogviewer,
            devices=devices,
            setraces=u"Данные Setrace на момент багрепорта:\n {}".format(traces_txt) if traces_txt else "",
            id=self.id,
            uuid=self.uuid,
            text=self.text,
        )

    def make_followers(self, author_as_follower):
        if author_as_follower and self.login.internal:
            return [self.login.internal]
        return []

    def make_attachments(self):
        def fix_filename(file_name):
            if file_name and file_name[0] == '/':
                file_name = file_name[1:]
            return file_name.replace('/', '.')

        attachments = []
        tmpdir = tempfile.mkdtemp()
        for name, value in self._files.iteritems():
            attachment = os.path.join(tmpdir, fix_filename(name))
            with open(attachment, 'wb') as f:
                f.write(value)
            attachments.append(attachment)
        return attachments

    def make_summary(self):
        return '{device_type}, {login}, {dt}'.format(
            device_type='+'.join(d.type for d in sorted(self._devices.itervalues())),
            login=self.login.make_signature(mailto=False) if self.login else '-',
            dt=datetime.datetime.fromtimestamp(self.timestamp),
        )

    def make_ticket(self, queue, followers, author_as_follower):
        return {
            'queue': queue,
            'type': {'name': 'Bug'},
            'summary': self.make_summary(),
            'description': self.make_description(),
            'followers': followers + self.make_followers(author_as_follower),
            'attachments': self.make_attachments(),
        }

    def __repr__(self):
        return u'BugReport from {uuid} at {timestamp} about "{text}"'.format(
            uuid=self.uuid,
            timestamp=datetime.datetime.fromtimestamp(self.timestamp),
            text=self.text,
        )


class Login(object):
    def __init__(self, external='', internal=''):
        self.external = external
        self.internal = internal

    def __nonzero__(self):
        return self.external != ''

    def make_signature(self, mailto=True):
        if self.external:
            login = '{}@yandex.ru'.format(self.external)
            if mailto:
                login = 'mailto:' + login
            if self.internal:
                login += ' ({}@)'.format(self.internal)
        else:
            login = ''
        return login


class Tvm2Blackbox(object):
    BLACKBOX_ALIAS_YANDEXOID = '13'
    BLACKBOX_TIMEOUT = 0.1
    BLACKBOX_URL = 'http://blackbox.yandex.net/blackbox?method=userinfo&userip=127.0.0.1&aliases=13&format=json&uid={}'

    TVM2_DESTINATION_BLACKBOX = '222'

    def __init__(self, client_id, secret):
        self._headers = {}
        self._login_cache = {}

        if not client_id or not secret:
            return

        from ticket_parser2.api.v1 import BlackboxClientId
        import tvm2

        tvm = tvm2.TVM2(
            client_id=client_id,
            blackbox_client=BlackboxClientId.Prod,
            secret=secret,
            destinations=(self.TVM2_DESTINATION_BLACKBOX,),
        )
        tvm_tickets = tvm.get_service_tickets(self.TVM2_DESTINATION_BLACKBOX)
        blackbox_ticket = tvm_tickets.get(self.TVM2_DESTINATION_BLACKBOX)
        self._headers['X-Ya-Service-Ticket'] = blackbox_ticket

    def get_login(self, account_id):
        if account_id in self._login_cache:
            return self._login_cache[account_id]
        login = None
        try:
            import requests
            response = requests.get(
                self.BLACKBOX_URL.format(account_id),
                headers=self._headers,
                timeout=self.BLACKBOX_TIMEOUT,
            )
            response.raise_for_status()
            data = response.json()
            logging.info('User %s data: %s', account_id, data)
            user = data['users'][0]
            login = Login(user['login'], user['aliases'].get(self.BLACKBOX_ALIAS_YANDEXOID, ''))
        except Exception:
            logging.exception('Getting logins for %s failed!', account_id)
        if login is not None:
            self._login_cache[account_id] = login
            return login
        return Login()


def get_bug_reports(rows, tmv2_blackbox, setrace_client, setrace_enabled):
    bug_reports = collections.defaultdict(BugReport)

    for row in rows:
        bug_report_id = row['BugReportId']

        if not bug_report_id or bug_report_id.startswith('music'):  # skip automatic reports
            continue

        if not row['AccountID']:  # skip bad reports
            continue

        raw_value = row['EventValue']

        try:
            value = json.loads(raw_value, strict=False)
        except ValueError:
            logging.exception('Parsing failed! %r', raw_value)
            continue

        assert bug_report_id == value['id']

        # support both old //logs/metrika-mobile-log and new //logs/appmetrica-yandex-events
        if isinstance(row['ReportEnvironment_Keys'], list):
            report_keys = row['ReportEnvironment_Keys']
        else:
            report_keys = json.loads(row['ReportEnvironment_Keys'])

        if isinstance(row['ReportEnvironment_Values'], list):
            report_vals = row['ReportEnvironment_Values']
        else:
            report_vals = json.loads(row['ReportEnvironment_Values'])

        report_environment = dict(zip(
            report_keys,
            report_vals,
        ))
        device_id = row.get('DeviceID', None)
        device_type = row.get('Model', None)

        if device_id is None or device_type is None:
            continue

        bug_report = bug_reports[bug_report_id]
        bug_report.id = bug_report_id
        bug_report.account_id = row['AccountID']
        bug_report.login = tmv2_blackbox.get_login(bug_report.account_id)
        t = row['BugReportMessage']
        if type(t) == str:
            t = t.decode('utf-8')
        elif t is None:
            t = u''
        else:
            t = t.encode('iso-8859-1', errors='replace').decode('utf-8')
        bug_report.text = t
        bug_report.timestamp = float(row['EventTimestamp'])

        report_environment['AppVersionName'] = row['AppVersionName']
        report_environment['SendTimestamp'] = datetime.datetime.fromtimestamp(float(row['SendTimestamp']))

        bug_report.add_device(Device(report_environment, device_id, device_type))

        if 'UUID' in row and row['UUID'] is not None and bug_report.uuid is None:
            bug_report.uuid = row['UUID']

        for logname in set(value.keys()) - {'id'}:
            logfile = value[logname]
            if logfile:
                try:
                    text = logfile.encode('iso-8859-1').decode('utf8').encode('utf8')
                except UnicodeEncodeError:
                    text = logfile.encode('utf-8')
                except UnicodeDecodeError:
                    text = logfile.encode('iso-8859-1', errors='replace')
                bug_report.add_file(logname, text)

    for bug_report in bug_reports.values():
        if bug_report.uuid is None:
            bug_report.uuid = hashlib.md5(bug_report.devices[0].id).hexdigest()

    if setrace_enabled:
        for br in bug_reports.values():
            br_timestamp = br.timestamp
            br_10_min_back = br_timestamp - datetime.timedelta(minutes=10).total_seconds()
            traces = {}
            try:
                for device in br.devices:
                    trace = setrace_client.get_alice_trace(device.id, br_10_min_back, br_timestamp)
                    traces["{}:{}".format(device.type, device.id)] = trace
                br._traces = traces
            except Exception:
                logging.error(u"failed to get setrace", exc_info=True)
    return bug_reports


def get_input_tables(yt_client, input_regexp, status_string_path):
    root, regexp = os.path.split(input_regexp)

    start_date = ''
    if status_string_path and yt_client.exists(status_string_path):
        start_date = yt_client.get(status_string_path)

    def path_filter(path):
        name = os.path.basename(path)
        return re.match(regexp, name) and name > start_date

    return sorted(str(i) for i in yt_client.search(root, node_type=['table'], path_filter=path_filter))


def main():
    parser = argparse.ArgumentParser()

    parser.add_argument('--yt_cluster', default='hahn')
    parser.add_argument('--yt_table_regexp', default='//home/quasar-dev/bug_reports_1h/.+')
    parser.add_argument('--status_string_path')
    parser.add_argument('--startrek_queue', default='QBRTEST')
    parser.add_argument('--startrek_followers', default='')
    parser.add_argument('--startrek_author_as_follower', action='store_true')
    parser.add_argument('--setrace_url', default='https://setrace.yandex-team.ru')
    parser.add_argument('--setrace_enabled', action='store_true')
    parser.add_argument('--tvm2_client_id')
    parser.add_argument('--dry_run', action='store_true')

    args = parser.parse_args()

    yt_cluster = args.yt_cluster
    yt_table_regexp = args.yt_table_regexp
    status_string_path = args.status_string_path
    startrek_queue = args.startrek_queue
    startrek_followers = args.startrek_followers.split(',') if args.startrek_followers else []
    startrek_author_as_follower = args.startrek_author_as_follower
    tvm2_client_id = args.tvm2_client_id
    setrace_url = args.setrace_url
    setrace_enabled = args.setrace_enabled
    dry_run = args.dry_run

    yt_token = os.environ['YT_TOKEN']
    startrek_token = os.environ['STARTREK_TOKEN']
    setrace_token = os.environ['SETRACE_TOKEN']
    tvm2_secret = os.environ.get('TVM2_SECRET')

    import startrek_client as startrek
    import yt.wrapper as yt
    from setrace_client import SetraceClient

    yt_client = yt.YtClient(yt_cluster, token=yt_token)
    setrace_client = SetraceClient(setrace_base_url=setrace_url, setrace_token=setrace_token)

    yt_input_tables = get_input_tables(yt_client, yt_table_regexp, status_string_path)
    logging.info('Input tables: %s', yt_input_tables)
    if not yt_input_tables:
        return
    yt_rows = itertools.chain.from_iterable(
        yt_client.read_table(yt_input_table, format='json')
        for yt_input_table in yt_input_tables
    )

    # use this for local runs
    # class MockTvm2Blackbox(object):
    #     def __init__(self, *ignore):
    #         pass
    #
    #     def get_login(self, account_id):
    #         return Login(str(account_id), str(account_id))

    bug_reports = get_bug_reports(yt_rows, Tvm2Blackbox(tvm2_client_id, tvm2_secret), setrace_client, setrace_enabled)
    startrek_client = startrek.Startrek(useragent=startrek_queue, token=startrek_token)

    for br in sorted(bug_reports.values(), key=lambda b: b.timestamp):
        raw_ticket = br.make_ticket(startrek_queue, startrek_followers, startrek_author_as_follower)
        if dry_run:
            logging.info("DRY RUN TICKET: {}".format(raw_ticket))
            continue
        ticket = startrek_client.issues.create(**raw_ticket)
        logging.info(ticket)

    if status_string_path:
        last_date = os.path.basename(yt_input_tables[-1])
        logging.info('Setting status %r', last_date)
        yt_client.set(status_string_path, last_date, recursive=True)


if __name__ == '__main__':
    logging.basicConfig(level=logging.DEBUG, format='%(asctime)s %(levelname)s: %(message)s')
    main()
