import enum
import logging
import sys
import typing as tp

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.common.types import notification as ctn

from . import validate

logger = logging.getLogger(__name__)
logger.setLevel(level=logging.INFO)
formatter = logging.Formatter("%(asctime)s %(levelname)s:%(message)s")
handler = logging.StreamHandler(stream=sys.stdout)
handler.setLevel(level=logging.INFO)
handler.setFormatter(formatter)
logger.addHandler(handler)


class ExpectedStatus(enum.Enum):
    BETA="beta"
    RELEASE_TEST="release_test"
    PRODUCTION="release"


STR_TO_STATUS = {
    ExpectedStatus.BETA.value: validate.EIssueStatusId.BETA,
    ExpectedStatus.RELEASE_TEST.value: validate.EIssueStatusId.RELEASE_TEST,
    ExpectedStatus.PRODUCTION.value: validate.EIssueStatusId.PRODUCTION,
}


class QuasarCheckReleaseStatus(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        owner = "QUASAR"
        description = "Checks quasar release issue status"
        startrek_token = sdk2.parameters.YavSecret(
            "YAV secret with startrek access",
            default="sec-01d2ffwrdbwyj37zkj4r8zegsn@ver-01f769hsrhrsdhyt4a3mg96gp6#robot-quasar-startrek-token"
        )
        release = sdk2.parameters.String("Release number")
        responsible = sdk2.parameters.String("Responsible staff login")
        queue = sdk2.parameters.String("Startrek queue to look up", default="ALICERELEASE")
        with sdk2.parameters.RadioGroup("Expected issue status") as expected_status:
            expected_status.values[ExpectedStatus.BETA.value] = expected_status.Value("Beta")
            expected_status.values[ExpectedStatus.RELEASE_TEST.value] = expected_status.Value("Release test")
            expected_status.values[ExpectedStatus.PRODUCTION.value] = expected_status.Value("Release")

        notifications = [
            sdk2.Notification(
                statuses=[
                    ctt.Status.SUCCESS,
                    ctt.Status.FAILURE,
                    ctt.Status.EXCEPTION,
                    ctt.Status.NO_RES,
                    ctt.Status.TIMEOUT,
                    ctt.Status.EXPIRED,
                ],
                recipients=["ohmikey@yandex-team.ru"],
                transport=ctn.Transport.EMAIL
            )
        ]

    def _get_startrek_issues(self):
        import startrek_client
        # type: (...) -> tp.List[validate.Issue]
        token = self.Parameters.startrek_token.data()[self.Parameters.startrek_token.default_key]
        client = startrek_client.Startrek(useragent="python", token=token)
        issues = []
        for issue in client.issues.find(
            filter=dict(
                queue=self.Parameters.queue,
                assignee=self.Parameters.responsible,
                tags=["release_number_{}".format(self.Parameters.release)]
            )
        ):
            logger.info("Issue found {}: {} {} {}".format(issue.key, issue.tags, issue.status.id, issue.status.display))
            issues.append(
                validate.Issue(
                    key=issue.key,
                    tags=issue.tags,
                    status=validate.IssueStatus(issue.status.id, issue.status.display)
                )
            )
        return issues

    def on_execute(self):
        import retry
        import startrek_client
        logger.info("Validation started")
        retry.api.retry_call(
            self._get_startrek_issues,
            exceptions=(startrek_client.exceptions.StartrekRequestError, startrek_client.exceptions.StartrekServerError),
            tries=5,
            delay=60,
            max_delay=60 * 25,
            backoff=5,
            logger=logger
        )
        issues = self._get_startrek_issues()
        try:
            validate.validate_startrek(
                issues,
                release=self.Parameters.release,
                responsible=self.Parameters.responsible,
                expected_status=STR_TO_STATUS[self.Parameters.expected_status]
            )
        except validate.error.ReleaseValidationError as e:
            self.server.notification(
                subject="Quasar release {} validation failed".format(self.Parameters.release),
                body=str(e),
                recipients=["{}@yandex-team.ru".format(self.Parameters.responsible)],
                transport=ctn.Transport.EMAIL,
            )
            raise
