import enum
import logging
import typing as tp

from . import error


logger = logging.getLogger(__name__)


class EIssueStatusId(enum.Enum):
    BETA="673"
    RELEASE_TEST="124"
    PRODUCTION="16"


class IssueTag(enum.Enum):
    STATION="quasar_yandexstation"
    STATION_MAX="quasar_yandexstation_2"


class IssueStatus:
    def __init__(
        self,
        id,       # type: str
        display,  # type: str
    ):
        self.id = id
        self.display = display


class Issue:
    def __init__(
        self,
        key,     # type: str
        tags,    # type: tp.List[str]
        status,  # type: IssueStatus
    ):
        self.key = key
        self.tags = tags
        self.status = status

    def __str__(self):
        # type: (...) -> str
        device = None
        if IssueTag.STATION.value in self.tags:
            device = "Station"
        elif IssueTag.STATION_MAX.value in self.tags:
            device = "Station.Max"
        else:
            device = "Unknown"
        return "{} status=({}, {}) device={}".format(self.key, self.status.id, self.status.display, device)


def _validate_issue(
    issue,            # type: Issue
    expected_status,  # type: EIssueStatusId
):
    # type: (...) -> bool
    return issue.status.id == expected_status.value


def _issues_to_str(
    issues,       # type: tp.List[Issue]
    filter=None,  # type: tp.Optional[tp.Callable[[Issue, int], bool]]
):
    # type: (...) -> str
    if filter is None:
        filter = lambda *_: True
    messages = []
    for idx, issue in enumerate(issues):
        if filter(issue, idx):
            messages.append(str(issue))
    return "\n".join(messages)


def _filter_tags(
    issues,  # type: tp.List[Issue]
):
    # type: (...) -> tp.List[Issue]
    expected_tags = set(e.value for e in IssueTag)
    return [i for i in issues if set(i.tags) & expected_tags]


def validate_startrek(
    issues,           # type: tp.List[Issue]
    release,          # type: str
    responsible,      # type: str
    expected_status,  # type: EIssueStatusId
):
    # type: (...) -> None
    issues_str = _issues_to_str(issues)
    logger.info("Issues:\n{}".format(issues_str))
    issues = _filter_tags(issues)
    if not issues:
        raise error.MissingIssueError("No issues found for filter: release={} responsible={}".format(release, responsible))
    if len(issues) != 2:
        raise error.IncorrectIssueCountError(
            "Expected 2 issues for filter: release={} responsible={}\nFound: {}".format(release, responsible, issues_str)
        )
    valid_statuses = [_validate_issue(issue, expected_status) for issue in issues]
    if not all(valid_statuses):
        def _filter(_, idx):
            # type: (Issue, int) -> bool
            return not valid_statuses[idx]
        message = _issues_to_str(issues, _filter)
        raise error.IncorrectIssueStatusError(
            "Following tickets have incorrect status:\n{}\nExpected status={}".format(
                message,
                expected_status.value
            )
        )
