# -*- coding: utf-8 -*-
import datetime
import logging

import requests

import sandbox.projects.release_machine.core.task_env as task_env
from sandbox.projects.quasar.platform import Platform
from sandbox import sdk2
from sandbox.sandboxsdk.channel import channel

logger = logging.getLogger(__name__)


class SmartDevicesCreateRegressTicket(sdk2.Task):
    MAD_HATTER_GROUP = 'WONDERLAND'
    MAD_HATTER_ST_TOKEN = 'env.STARTREK_OAUTH_TOKEN'
    MAD_HATTER_TESTPALM_TOKEN = 'env.TESTPALM_OAUTH_TOKEN'

    SUPPORTED_PLATFORMS = {
        Platform.YANDEXMICRO,
        Platform.YANDEXMIDI,
        Platform.YANDEXMINI,
        Platform.YANDEXMINI_2,
        Platform.YANDEXSTATION,
        Platform.YANDEXSTATION_2,
        Platform.JBL_LINK_MUSIC,
        Platform.JBL_LINK_PORTABLE,
        Platform.YANDEXMODULE_2,
    }

    @staticmethod
    def get_testpalm_tag(value):
        if value == Platform.YANDEXMICRO:
            return 'station lite'
        if value == Platform.YANDEXMIDI:
            return 'midi'
        if value == Platform.YANDEXMINI:
            return 'station mini'
        if value == Platform.YANDEXMINI_2:
            return 'station_mini_2'
        if value == Platform.YANDEXSTATION:
            return 'quasar_station_1'
        if value == Platform.YANDEXSTATION_2:
            return 'quasar_station_2'
        if value == Platform.JBL_LINK_MUSIC or value == Platform.JBL_LINK_PORTABLE:
            return 'JBL'
        if value == Platform.YANDEXMODULE_2:
            return 'module_2'
        return 'unsupported'

    class Parameters(sdk2.Task.Parameters):
        release_ticket_key = sdk2.parameters.String('Release ticket key', default="ALICERELEASE-XXX", required=True)
        assessors_queue = sdk2.parameters.String('Assessors queue', default="ALICEASSESSORS", required=True)
        target_platform = sdk2.parameters.String('Target platform', required=True)
        testpalm_project = sdk2.parameters.String('Testpalm project', default="alice", required=True)
        build_variants = sdk2.parameters.List('Build variants', required=True)

        with sdk2.parameters.Output:
            regress_ticket_key = sdk2.parameters.String("Regress ticket for release")

    class Requirements(task_env.StartrekRequirements):
        pass

    def on_execute(self):
        release_ticket_key = self.Parameters.release_ticket_key
        assessors_queue = self.Parameters.assessors_queue
        target_platform = self.Parameters.target_platform.lower()
        testpalm_project = self.Parameters.testpalm_project
        build_variants = self.Parameters.build_variants

        logger.info("Regress ticket requested for {platform} release {release_ticket}".format(
            platform=target_platform,
            release_ticket=release_ticket_key
        ))

        if target_platform not in self.SUPPORTED_PLATFORMS:
            logger.info("Platform is not supported: " + target_platform)
            return None

        st_token, testpalm_token = self.get_tokens()

        st_client = StartrekClient(st_token, useragent="smart_devices")
        regress_key = st_client.check_regress_ticket_exists(assessors_queue, target_platform, release_ticket_key)
        if not regress_key:
            testpalm_client = TestpalmClient(testpalm_token)
            testpalm_tag = self.get_testpalm_tag(target_platform)
            testpalm_runs = testpalm_client.create_runs(testpalm_project, testpalm_tag, release_ticket_key)

            regress_key = st_client.create_regress_ticket(assessors_queue, target_platform,
                                                          release_ticket_key, testpalm_runs, build_variants)

        self.Parameters.regress_ticket_key = regress_key

    def get_tokens(self):
        logger.info(
            "Receiving st {st} and testpalm {testpalm} tokens of group {group}".format(
                st=self.MAD_HATTER_ST_TOKEN,
                testpalm=self.MAD_HATTER_TESTPALM_TOKEN,
                group=self.MAD_HATTER_GROUP,
            )
        )

        try:
            st_token = channel.task.get_vault_data(self.MAD_HATTER_GROUP, self.MAD_HATTER_ST_TOKEN)
            testpalm_token = channel.task.get_vault_data(self.MAD_HATTER_GROUP, self.MAD_HATTER_TESTPALM_TOKEN)
        except Exception as error:
            raise Exception("Unable to get tokens, reason: {error}".format(error=error))

        logger.info("Tokens received successfully")

        return st_token, testpalm_token


class TestpalmClient(object):
    TESTPALM_API_URL = "https://testpalm-api.yandex-team.ru"
    TESTPALM_URL = "https://testpalm.yandex-team.ru"

    def __init__(self, testpalm_token):
        self.testpalm_token = testpalm_token

    def create_runs(self, project, tag, release_ticket_key):
        logger.info(
            "Creating runs in project {project} for ticket {release_ticket_key} and tag {tag}".format(
                project=project,
                release_ticket_key=release_ticket_key,
                tag=tag
            )
        )

        suites = []
        created_runs = []

        try:
            suites = self.get_suites_by_tag(project, tag)
        except Exception as error:
            logger.error("Unable to get suites by tag {tag}, reason: {error}".format(tag=tag, error=error))

        for suite in suites:
            try:
                title = self.generate_run_title(suite.get("title"), release_ticket_key, tag, datetime.datetime.now())
                run = self.create_run(project, title, suite, release_ticket_key, suite.get("tags"))
                if run is not None:
                    created_runs.append(run)
            except Exception as error:
                logger.error("Unable to create regress run, reason: {error}".format(error=error))

        logger.info(
            "Created runs {created_runs}".format(
                created_runs=created_runs
            )
        )
        return created_runs

    def get_suites_by_tag(self, project, tag):
        logger.info(
            "Getting suites in project {project} for tag {tag}".format(
                project=project,
                tag=tag
            )
        )

        url = "{base_url}/testsuite/{project}".format(
            base_url=self.TESTPALM_API_URL,
            project=project,
        )
        headers = self.generate_header(self.testpalm_token)
        params = {
            "include": "id,title,tags",
            "expression": '''{{"type": "CONTAIN", "key": "tags", "value": "{tag}" }}'''.format(tag=tag)
        }
        response = requests.get(url, headers=headers, params=params)
        if response.status_code != 200:
            raise Exception(
                "Unable to get testpalm suites response, code: {code}, body: {body}".format(
                    code=response.status_code,
                    body=response.text,
                )
            )
        suites = response.json()

        logger.info(
            "Received suites {suites}".format(
                suites=suites
            )
        )
        return suites

    def create_run(self, project, title, suite, ticket_key, tags):
        logger.info(
            "\tCreating run {title} in project {project} for suite {suite_id}".format(
                title=title,
                project=project,
                suite_id=suite.get("id")
            )
        )

        url = "{base_url}/testrun/{project}/create".format(
            base_url=self.TESTPALM_API_URL,
            project=project,
        )
        headers = self.generate_header(self.testpalm_token)
        params = {
            "include": "id,title",
        }
        payload = {
            "tags": tags,
            "title": title,
            "testSuite": {
                "id": suite.get("id")
            },
            "ignoreSuiteOrder": False,
            "parentIssue": {
                "id": ticket_key,
                "trackerId": "Startrek"
            }
        }
        response = requests.post(url, json=payload, params=params, headers=headers)
        if response.status_code != 200:
            logger.error(
                "Unable to create testpalm run, code: {code}, body: {body}".format(
                    code=response.status_code,
                    body=response.text,
                )
            )
            return None
        run = next(iter(response.json()), {})
        run["url"] = "{base_url}/{project}/testrun/{id}".format(
            base_url=self.TESTPALM_URL,
            project=project,
            id=run.get('id')
        )
        run["key"] = "testrun/{project}/{id}".format(
            project=project,
            id=run.get('id')
        )
        run["tags"] = suite.get("tags")

        logger.info(
            "\tCreated run {run}".format(
                run=run
            )
        )
        return run

    @staticmethod
    def generate_header(token):
        return {
            'Authorization': 'OAuth {}'.format(token),
            'Content-Type': 'application/json',
        }

    @staticmethod
    def generate_run_title(suite_title, ticket_key, tag, date):
        run_title = "{title}: {ticket_key} {tag} {date}".format(
            title=suite_title[:suite_title.find(']') + 1],
            ticket_key=ticket_key,
            tag=tag,
            date=date.strftime("%d.%m"),
        )
        return run_title


class StartrekClient(object):
    def __init__(self, st_token, useragent=None):
        if not useragent:
            useragent = "smart_devices"
        from startrek_client import Startrek
        self.client = Startrek(useragent=useragent, token=st_token)

    def check_regress_ticket_exists(self, assessors_queue, platform, release_ticket_key):
        logger.info("Looking for assessors ticket for {ticket_key}".format(ticket_key=release_ticket_key))

        title = self.create_regress_title(platform, release_ticket_key)
        try:
            release_ticket = None
            try:
                release_ticket = self.client.issues[release_ticket_key]
            except Exception as error:
                raise Exception("Unable to get release ticket {release_ticket_key}: {error}".format(
                    release_ticket_key=release_ticket_key,
                    error=error
                ))
            for link in release_ticket.links:
                try:
                    linked_ticket = link.object
                    if linked_ticket.queue.name == assessors_queue and linked_ticket.summary == title:
                        ticket_key = linked_ticket.key
                        logger.info("Found existing regress ticket {key}".format(key=ticket_key))

                        return ticket_key
                except Exception as error:
                    logger.error("Unable to get linked ticket {link}: {error}".format(
                        link=link,
                        error=error
                    ))
        except Exception as error:
            raise Exception("Unable to check for existing ticket: {error}".format(error=error))

        logger.info("Assessors ticket not found in queue {assessors_queue}".format(assessors_queue=assessors_queue))

        return None

    def create_regress_ticket(self, assessors_queue, platform, release_ticket_key, testpalm_runs, build_variants):
        logger.info("Creating assessors ticket for {ticket_key}".format(ticket_key=release_ticket_key))

        title = self.create_regress_title(platform, release_ticket_key)
        description = self.create_assessors_description(platform, release_ticket_key, testpalm_runs, build_variants)
        try:
            ticket = self.client.issues.create(
                queue=assessors_queue,
                summary=title,
                description=description,
                tags='cases',
                type='task',
                links={
                    "relationship": "is subtask for",
                    "issue": release_ticket_key
                },
            )
        except Exception as error:
            raise Exception("Unable to create ticket, reason: {error}".format(error=error))

        logger.info(
            "Created ticket {ticket}".format(
                ticket=ticket.key
            )
        )
        return ticket.key

    @staticmethod
    def create_regress_title(platform, release_ticket_key):
        return (
            "Тестирование релиза {platform} {release_ticket_key}"
        ).format(platform=platform, release_ticket_key=release_ticket_key)

    @staticmethod
    def generate_run_line(run):
        title = run.get("title")
        return "{title} {url}".format(
            title=title[:title.find(']') + 1],
            url=run.get("url")
        )

    @staticmethod
    def create_assessors_description(platform, release_ticket_key, testpalm_runs, build_variants):
        text_header = (
            '====Релиз {platform}====\n'
            'Релизный тикет {release_ticket_key}\n\n'
        ).format(platform=platform, release_ticket_key=release_ticket_key)

        opt_config = ''
        if build_variants is not None and len(build_variants) > 1:
            opt_config = (
                '<{Раскрыть конфиг\n'
                'CONFIG_PLACEHOLDER\n'
                '}>\n\n'
            )

        text_config = ''
        if platform != Platform.YANDEXMODULE_2:
            text_config = (
                '====Конфиг:====\n'
                '{opt_config}'
                'Добавить флаги в массив "experiments":\n'
                '+ флаг для включения режима "отправь баг-репорт": "debug_mode"\n'
                '+ флаг, отключающий шальные эксперименты алисы: "only_100_percent_flags"\n'
                '+ флаг "enable_full_rtlog"\n\n'
                '+ в корень конфига добавить флаг "no_tests": true\n\n'
                'Как получить конфиг рассказано здесь: https://wiki.yandex-team.ru/alicetesting/assessors-and-alice/assessors-and-station/#kakpoluchitprodversiiproshivokiprodkonfigi\n\n'
            ).format(opt_config=opt_config)

        firmware_placeholders = '====Прошивки:====\n'
        for build_variant_name in build_variants:
            if build_variant_name is None:
                firmware_placeholders += 'VERSION_PLACEHOLDER\n\n'
            else:
                firmware_placeholders += '{}: VERSION_PLACEHOLDER\n\n'.format(build_variant_name.upper())

        text_after_firmware = ''
        if platform == Platform.YANDEXMODULE_2:
            text_after_firmware = (
                'Версию прошивку из ссылки указываем в Forced Firmware в космодроме\n'
                'Пример:\n'
                'https://jing.yandex-team.ru/files/osennikovak/2021-11-08T11:30:25Z.e138747.png\n\n'
                '====Для того чтобы приступить к тестрану:====\n'
                '1) Выставить на устройстве дефолтный конфиг.\n'
                '2) Убедитесь, что все обновления прилетели и установились.\n'
                'Системные настройки -- Об устройстве -- Проверить обновление системы (Установить, если версия не последняя)\n'
                'Системные настройки -- Приложения -- Все приложения -- YandexTVHome (Должна быть версия 2.113.*)\n\n'
                'Тандем проходить с микро, прошивка и конфиг -- прод; кейсы в ране тандема, где нужна большая колонка, проходить со ст2, прошивка и конфиг -- прод\n\n'
            )

        text_runs = ''
        if testpalm_runs:
            text_runs = (
                '====Список ранов====\n'
                '<['
                '{runs}'
                ']>\n\n'
            ).format(runs="\n".join(map(StartrekClient.generate_run_line, testpalm_runs)))

        text_instruction = (
            '====Инструкция по тестированию====\n'
            'https://wiki.yandex-team.ru/alicetesting/assessors-and-alice/\n\n'
        )
        if platform == Platform.YANDEXMODULE_2:
            text_instruction += (
                'Инструкция по тестированию ТВ/Модуля-2 - https://wiki.yandex-team.ru/alicetesting/assessors-and-alice/asessorsintv/\n'
                'Глоссарий по ТВ/Модуля-2 - https://wiki.yandex-team.ru/alicetesting/assessors-and-alice/asessorsintv/Глоссарий-для-ТВ/\n\n'
            )

        return (
            '{text_header}'
            '{text_config}'
            '{firmware_placeholders}'
            '{text_after_firmware}'
            '{text_runs}'
            '{text_instruction}'
        ).format(
            text_header=text_header,
            text_config=text_config,
            firmware_placeholders=firmware_placeholders,
            text_after_firmware=text_after_firmware,
            text_runs=text_runs,
            text_instruction=text_instruction
        )
