# coding=utf-8
import json
import logging
from datetime import datetime

from sandbox import sdk2, common
from sandbox.sandboxsdk.process import run_process
from sandbox.sdk2.service_resources import SandboxTasksBinary


class QuasarDatasyncUploader(sdk2.Task):
    """
    https://st.yandex-team.ru/CHEMODAN-62427
    https://st.yandex-team.ru/QUASAR-3223

    Exports users devices to DataSync.
    """

    configs = {"yandexstation": {"name": u"Яндекс Станция",
                                 "can_have_screen": True},
               "yandexmodule": {"name": u"Яндекс Модуль",
                                "can_have_screen": True},
               "yandexstation_2": {"name": u"Яндекс Станция Макс",
                                   "can_have_screen": True}}

    default_config = {"name": u"Колонка от Яндекс",
                      "can_have_screen": False}

    class Requirements(sdk2.Requirements):
        pass

    class Parameters(sdk2.Parameters):
        DryRun = sdk2.parameters.Bool('Do not write data to datasync', default=False)
        UseLastBinary = sdk2.parameters.Bool('Use last binary archive', default=True)
        with UseLastBinary.value[True]:
            with sdk2.parameters.RadioGroup("Binary release type") as ReleaseType:
                ReleaseType.values.stable = ReleaseType.Value('stable', default=True)
                ReleaseType.values.test = ReleaseType.Value('test')

        DataSyncUrl = sdk2.parameters.String("Datasync api url", default="https://datasync.yandex.net:8443")
        DatasyncTvmId = sdk2.parameters.Integer("Datasync tvm id", default=107)
        TvmSelfId = sdk2.parameters.Integer("Self tvm id", default=2011246)
        TvmSelfSecretName = sdk2.parameters.String("Self tvm secret name", default="quasar-datasync-uploader-tvm-secret")

        DBName = sdk2.parameters.String('Quasar db name', default="quasar_backend")
        DBUser = sdk2.parameters.String('Quasar db user', default="quasar_ro")
        DBPort = sdk2.parameters.String('Quasar db port', default="6432")
        DBHost = sdk2.parameters.String('Quasar db host', default="vla-9ednmkhlmhldloz9.db.yandex.net,man-w097xhoothwhma3d.db.yandex.net,sas-70qznnvkzb8zyhdi.db.yandex.net")

    @staticmethod
    def prepare_certs():
        run_process(
            ["curl", "https://crls.yandex.net/allCAs.pem", "-o", "allCAs.pem"],
            work_dir=".",
            log_prefix="curl"
        )

    def on_save(self):
        if self.Parameters.UseLastBinary:
            resource = SandboxTasksBinary.find(
                attrs=dict(
                    target='sandbox/projects/quasar/datasync_uploader',
                    release=self.Parameters.ReleaseType or 'stable')).first()
            self.Requirements.tasks_resource = resource.id if resource else None
        else:
            self.Requirements.tasks_resource = None

        if self.Parameters.ReleaseType == "stable":
            self.Parameters.DataSyncUrl = "https://datasync.yandex.net:8443"
            self.Parameters.TvmSelfId = 2011246
            self.Parameters.DatasyncTvmId = 107
            self.Parameters.TvmSelfSecretName = "quasar-datasync-uploader-tvm-secret"
        elif self.Parameters.ReleaseType == "test":
            self.Parameters.DataSyncUrl = "https://dataapi.dst.yandex.net:8443"
            self.Parameters.TvmSelfId = 2002639
            self.Parameters.DatasyncTvmId = 2000962
            self.Parameters.TvmSelfSecretName = "quasar-test-backend-tvm-secret"

    def on_execute(self):
        from uploader import DatasyncUploader
        from pytz import utc

        self.prepare_certs()

        pgconn = self.get_pgaas_connection()
        service_ticket = self.get_service_ticket()
        with DatasyncUploader(base_url=self.Parameters.DataSyncUrl, service_ticket=service_ticket, dry_run=self.Parameters.DryRun, batch_size=1000) as datasync_uploader:
            with pgconn.cursor() as cur:
                cur.execute('SELECT dc.json_config, ad.device_id, ad.account_id, p.name FROM account_device ad ' +
                                     'LEFT JOIN platform p ON ad.platform_id=p.id ' +
                                     'LEFT JOIN device_config dc ON ad.device_id = dc.device_id')
                for (json_config_raw, device_id, uuid, platform) in cur:
                    json_config = json.loads(json_config_raw) if json_config_raw else dict()
                    defaults = QuasarDatasyncUploader.configs.get(platform, QuasarDatasyncUploader.default_config)
                    name = json_config['name'] if 'name' in json_config else defaults['name']

                    updated = utc.localize(datetime.utcnow())
                    modified_at = updated.strftime(DatasyncUploader.fmt)

                    screen_capable = defaults['can_have_screen']
                    screen_present = defaults['can_have_screen']

                    firmware = ""

                    datasync_uploader.add_datasync_entry(uuid=uuid,
                                                         name=name,
                                                         device_id=device_id,
                                                         platform=platform,
                                                         firmware_version=firmware,
                                                         modified_at=modified_at,
                                                         screen_capable=screen_capable,
                                                         screen_present=screen_present)

                    logging.info(u"name={}, device={}, uuid={}, platform={}, version={}, modified_at={}, screen_capable={}, screen_present={}".format(
                        name,
                        device_id,
                        uuid,
                        platform,
                        firmware,
                        modified_at,
                        screen_capable,
                        screen_present))

    def get_connection_string(self, password):
        return "dbname={dbname} user={dbuser} port={dbport} host={dbhost} sslmode=verify-full password={dbpassword} sslrootcert=allCAs.pem".format(
            dbname=self.Parameters.DBName,
            dbuser=self.Parameters.DBUser,
            dbport=self.Parameters.DBPort,
            dbhost=self.rearrange_db_hosts(self.Parameters.DBHost),
            dbpassword=password)

    def get_pgaas_connection(self):
        password = sdk2.Vault.data("QUASAR", "robot-quasar-pgaas-backend-password")
        import psycopg2
        return psycopg2.connect(self.get_connection_string(password))

    def get_service_ticket(self):
        from ticket_parser2.api.v1 import (
            TvmClient,
            TvmApiClientSettings,
        )

        secret = sdk2.Vault.data("QUASAR", self.Parameters.TvmSelfSecretName)
        tvm_settings = TvmApiClientSettings(
            self_client_id=self.Parameters.TvmSelfId,
            self_secret=secret,
            enable_service_ticket_checking=True,
            dsts=dict(datasync=self.Parameters.DatasyncTvmId))

        TVM = TvmClient(tvm_settings)
        service_ticket = TVM.get_service_ticket_for("datasync")

        if service_ticket is None:
            raise Exception("Unable to get service ticket from quasar-datasync-job({}) to datasync-api({})"
                            .format(QuasarDatasyncUploader.QUASAR_DATASYNC_JOB_TVM_ID,
                                    QuasarDatasyncUploader.DATASYNC_TVM_ID))
        return service_ticket

    @staticmethod
    def rearrange_db_hosts(hosts):
        """
        Should use closest db host. Or any
        :param hosts: hosts string aka 'sas-host1.yandex.net,man-host2.yandex.net'
        :return:
        """
        dc = common.config.Registry().this.dc
        tail = []
        head = []
        for host in hosts.split(','):
            if host.startswith(dc):
                head.append(host)
            else:
                tail.append(host)

        return ",".join(head + tail)
