import logging
import json
import time
import traceback
from urlparse import urljoin

import grequests


logging.getLogger(__name__).setLevel(logging.INFO)


class RetryableException(Exception):
    pass


class FatalException(Exception):
    pass


class DatasyncUploader(object):
    """
    uploads data to datasync with a batches. Batch size set to batch_size.
    """

    # like '2019-04-05T09:44:47Z'
    fmt = '%Y-%m-%dT%H:%M:%SZ'

    def __init__(self, base_url, service_ticket, dry_run=False, batch_size=50):
        self.base_url = base_url
        self.service_ticket = service_ticket
        self.entries = []
        self.dry_run = dry_run
        self.batch_size = batch_size

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type:
            logging.error(traceback.format_exception(exc_type, exc_val, exc_tb))
            return

        if self.entries:
            self.flush()

    def add_datasync_entry(self,
                           uuid,
                           device_id,
                           **datasync_fields):
        """

        :param uuid: user passport uid
        :param device_id:
        :param datasync_fields: dict that will be pushed to datasync
        :return:
        """
        # There is a batch ruchka but for different users it doesn't provide any significant profit
        self.entries.append(dict(relative_url="/v1/personality/profile/yandex_io/devices/{device_id}".format(device_id=device_id),
                                 data=datasync_fields,
                                 device_id=device_id,
                                 headers={"X-Ya-Service-Ticket": self.service_ticket, "X-Uid": str(uuid)}))
        self.flush_maybe()

    def flush_maybe(self):
        if len(self.entries) >= self.batch_size:
            self.flush()

    def flush(self):
        items = []

        for entry in self.entries:
            data = entry['data']
            items.append({
                "headers": entry['headers'],
                "data": data,
                "relative_url": entry['relative_url']
            })
        if not self.dry_run:
            tries = 3
            delay = 2

            while tries:
                try:
                    items_to_retry, errors = self.do_plain_request(items)
                    if not items_to_retry:
                        # everything is ok
                        break
                    tries -= 1
                    if tries:
                        time.sleep(delay)

                        # retry failed items
                        items = items_to_retry
                        continue

                    # no retries left
                    raise FatalException(u'All attempts to write to datasync were failed. Errors was: ' + u'-------------------NEXT ERROR-------------\n'.join(errors))

                except RetryableException:
                    tries -= 1
                    if tries:
                        time.sleep(delay)
                        # retry everything
                        continue
                    # shutdown
                    logging.exception("All attempts to write to datasync were failed")
                    raise

        logging.info('Device ids processed: {}'.format(', '.join(map(lambda e: e['device_id'], self.entries))))
        self.entries = []

    def do_plain_request(self, items):

        def request(item):
            url = urljoin(self.base_url, item['relative_url'])
            data = item['data']
            headers = item['headers']
            headers.update({'Content-Type': 'application/json'})
            logging.debug("Using plain control v1/personality/profile/yandex_io/devices/ :")
            logging.debug("url:{}".format(url))
            logging.debug("json:{}".format(json.dumps(data)))

            return grequests.put(url,
                                 json=data,
                                 headers=headers)

        pending = []
        for item in items:
            pending.append(request(item))

        def on_exception(_, e):
            raise RetryableException(u"exception during request. Exception: {}".format(unicode(e)))

        reqs = grequests.map(pending, exception_handler=on_exception)

        requests_with_items = zip(items, reqs)

        errors = []
        error_items = []
        for item, resp in requests_with_items:
            if resp.status_code != 200:
                if resp.status_code == 500:
                    try:
                        json_ = resp.json()
                        if "error" in json_ and "name" in json_ and json_["name"] == "user-read-only":
                            # user is moving to antother db cluster and will be available in 1.5 minutes
                            logging.warning(u"Put to datasync failed due to user is read-only. UID:{}".format(resp.request.headers.get("X-Uid")))
                            continue
                    except ValueError:
                        logging.error(u"Unable to parse datasync response")

                errors.append(u"Put Request to datasync failed. Code: {}. Body: {}".format(resp.status_code, unicode(resp.content, errors="replace")))
                error_items.append(item)

        if errors:
            return error_items, errors

        return [], []
