import sandbox.common as sandbox_common
import sandbox.projects.common.binary_task as binary_task
import sandbox.sdk2 as sdk2
import sandbox.projects.quasar.utils.vcs as quasar_vcs

import json
import logging
import os
import shutil
import subprocess
import urlparse


def safe_run_repo_tool(repo_args, with_output=False, **kwargs):
    import retrying

    @retrying.retry(stop_max_attempt_number=3, wait_exponential_multiplier=60000)
    def _run_repo_tool():
        if with_output:
            return subprocess.check_output(repo_args, **kwargs)
        else:
            subprocess.check_call(repo_args, **kwargs)

    try:
        return _run_repo_tool()
    except subprocess.CalledProcessError as e:
        raise sandbox_common.errors.TaskError('repo exited with code {}, {}'.format(e.returncode, e.message))


class ExternalRepositoryTracker(binary_task.LastBinaryTaskRelease, sdk2.Task):
    MANIFEST_FILENAME = 'manifest.xml'

    class Parameters(binary_task.LastBinaryReleaseParameters):
        ssh_user = sdk2.parameters.String('Ssh user name')
        ssh_key_vault_owner = sdk2.parameters.String('Sandbox vault item owner for ssh key')
        ssh_key_vault_name = sdk2.parameters.String('Sandbox vault item name for ssh key')
        dry_run = sdk2.parameters.Bool('Do no commit state to arcadia', default=False)
        arcadia_url = sdk2.parameters.ArcadiaUrl('Path in arcadia (arcadia:/arc/trunk/arcadia/path_in_arcadia)')
        arcadia_urls = sdk2.parameters.List(
            'Paths in arcadia to repositiry config dir (arcadia:/arc/trunk/arcadia/path_in_arcadia)')
        _container = sdk2.parameters.Container(
            'lxc container with repo installed',
            default_value=quasar_vcs.Containers.REPO_CONTAINER_ID,
            platform='linux_ubuntu_18.04_bionic',
            required=True,
        )

    def on_execute(self):
        super(ExternalRepositoryTracker, self).on_execute()

        with sdk2.ssh.Key(
            self,
            key_owner=self.Parameters.ssh_key_vault_owner,
            key_name=self.Parameters.ssh_key_vault_name
        ):
            arcadia_repository_state_dirname = os.path.basename(str(self.Parameters.arcadia_url).rstrip('/'))
            sdk2.svn.Arcadia.checkout(
                url=self.Parameters.arcadia_url,
                path=arcadia_repository_state_dirname,
            )
            repository_config_filename = os.path.join(arcadia_repository_state_dirname, 'repository.json')
            with open(repository_config_filename) as repository_config_file:
                repository_config = json.load(repository_config_file)

            if repository_config['vcs'] == quasar_vcs.VCS.REPO:
                repository_state_update = self.repo_state_extractor(
                    arcadia_repository_state_dirname, repository_config
                )

            if repository_state_update['need_commit']:
                commited_paths = set()
                for subpath, commit_dict in repository_state_update['updates'].items():
                    logging.info(sdk2.svn.Arcadia.status(arcadia_repository_state_dirname))
                    logging.info(commit_dict['message'])

                    if self.Parameters.dry_run:
                        continue

                    commit_path = os.path.join(arcadia_repository_state_dirname, subpath)
                    if commit_dict['new_branch']:
                        commit_path = arcadia_repository_state_dirname
                        commited = False
                        for path_part in subpath.split(os.path.sep):
                            commit_path = os.path.join(commit_path, path_part)
                            if commit_path in commited_paths:
                                commited = True
                                break
                            else:
                                try:
                                    sdk2.svn.Arcadia.add(commit_path)
                                    sdk2.svn.Arcadia.commit(
                                        path=commit_path,
                                        message='{} SKIP_CHECK'.format(commit_dict['message']),
                                        user=self.Parameters.ssh_user,
                                    )
                                    commited_paths.add(commit_path)
                                    commited = True
                                    break
                                except sdk2.svn.SvnError:
                                    logging.exception('Svn operation failed')

                        if not commited:
                            raise Exception('Something went wrong and {} was not commited'.format(commit_path))
                    else:
                        sdk2.svn.Arcadia.commit(
                            path=commit_path,
                            message='{} SKIP_CHECK'.format(commit_dict['message']),
                            user=self.Parameters.ssh_user,
                        )

    def get_checkout_url(self, repository_url):
        split = urlparse.urlsplit(repository_url)
        if self.Parameters.ssh_user:
            netloc = split.netloc.split('@')[-1]
            split = split._replace(netloc='{}@{}'.format(self.Parameters.ssh_user, netloc))

        return split.geturl()

    def repo_checkout(self, repo_url, branch):
        with sdk2.helpers.ProcessLog(self, logger='repo_init') as pl:
            repo_init_list = [
                'repo', 'init',
                '-u', self.get_checkout_url(repo_url),
                '-b', branch,
            ]

            logging.info('Running %s', ' '.join(repo_init_list))
            safe_run_repo_tool(repo_init_list, stdout=pl.stdout, stderr=pl.stderr)
        with sdk2.helpers.ProcessLog(self, logger='repo_sync') as pl:
            repo_sync_list = [
                'repo', 'sync',
                '-j', '8',
                '--force-sync',
                '--retry-fetches', '3',
            ]
            logging.info('Running %s', ' '.join(repo_sync_list))
            safe_run_repo_tool(repo_sync_list, stdout=pl.stdout, stderr=pl.stderr)

    def diff_manifests(self, manifest1, manifest2):
        with sdk2.helpers.ProcessLog(self, logger='repo_diffmanifests') as pl:
            repo_diffmanifests_list = ['repo', 'diffmanifests', str(manifest1), str(manifest2), '--no-color']
            logging.info('Running %s', ' '.join(repo_diffmanifests_list))
            return safe_run_repo_tool(repo_diffmanifests_list, with_output=True, stderr=pl.stderr)

    @staticmethod
    def get_arcadia_manifest_data(arcadia_manifest_filename):
        if not os.path.exists(arcadia_manifest_filename):
            branch_dirname = os.path.dirname(arcadia_manifest_filename)
            if not os.path.exists(branch_dirname):
                os.makedirs(branch_dirname)
            with open(arcadia_manifest_filename, 'w'):
                pass
        with open(arcadia_manifest_filename) as arcadia_manifest_file:
            arcadia_manifest_data = arcadia_manifest_file.read()

        return arcadia_manifest_data

    def repo_state_extractor(self, arcadia_repository_state_dirname, repository_config):  # noqa
        repo_state_update = {'need_commit': False, 'updates': {}}
        for branch, branch_config in repository_config['tracked_branches'].items():
            self.repo_checkout(repository_config['url'], branch)
            with sdk2.helpers.ProcessLog(self, logger='repo_manifest') as pl:
                repo_manifest_list = ['repo', 'manifest', '-r', '-o', self.MANIFEST_FILENAME]
                logging.info('Running %s', ' '.join(repo_manifest_list))
                safe_run_repo_tool(repo_manifest_list, stdout=pl.stdout, stderr=pl.stderr)
            with open(self.MANIFEST_FILENAME) as manifest_file:
                manifest_data = manifest_file.read()
                logging.debug(manifest_data)

            arcadia_manifest_filename = os.path.join(arcadia_repository_state_dirname, branch, self.MANIFEST_FILENAME)
            arcadia_manifest_data = self.get_arcadia_manifest_data(arcadia_manifest_filename)

            if manifest_data != arcadia_manifest_data:
                repo_state_update['need_commit'] = True
                if arcadia_manifest_data:
                    commit_message = self.diff_manifests(arcadia_manifest_filename, self.MANIFEST_FILENAME)

                    repo_state_update['updates'][branch] = {
                        'new_branch': False,
                    }
                else:
                    commit_message = 'Initial commit'
                    repo_state_update['updates'][branch] = {
                        'new_branch': True,
                    }

                if branch_config.get('mergeto'):
                    commit_message += '\n[mergeto:{}]'.format(branch_config['mergeto'])
                repo_state_update['updates'][branch]['message'] = commit_message

                shutil.copyfile(self.MANIFEST_FILENAME, arcadia_manifest_filename)

        return repo_state_update
