# coding: utf-8

from __future__ import unicode_literals

import calendar
import logging
import tempfile

import requests

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.common import binary_task

EVENTS_SCHEMA = [
    {'name': 'timestamp', 'type': 'double', 'required': True, 'sort_order': 'ascending'},
    {'name': 'device_uuid', 'type': 'string', 'required': True},
    {'name': 'chamber_id', 'type': 'string', 'required': True},
    {'name': 'description', 'type': 'string', 'required': True},
    {'name': 'payload', 'type': 'string', 'required': True},
    {'name': 'event_type', 'type': 'string', 'required': False},
    {'name': 'device_model', 'type': 'string', 'required': False},
]


class FetchJanglesEvents(binary_task.LastBinaryTaskRelease, sdk2.Task):
    # upload to sandbox via ./fetch_jangles_events upload --attr 'name=FetchJanglesEvents' and release resource
    TASKS_RESOURCE_NAME = 'FetchJanglesEvents'
    LOCAL_PORT = 8080

    class Requirements(sdk2.Requirements):
        cores = 1  # vCores
        ram = 1024  # Mb
        disk_space = 2 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    # noinspection DuplicatedCode
    class Parameters(binary_task.LastBinaryReleaseParameters):

        with sdk2.parameters.Group('SSH'):
            user = sdk2.parameters.String('SSH user', default='yandex')
            # noinspection SpellCheckingInspection
            host = sdk2.parameters.String('SSH host', default='fenda.ext.eine.yandex.net')
            port = sdk2.parameters.Integer('SSH port', default=1222)

        with sdk2.parameters.Group('Filters'):
            after_timestamp = sdk2.parameters.Float('after timestamp', default=0.0)
            device_model = sdk2.parameters.String('Device model', default='yandex_station_pro')
            data_limit = sdk2.parameters.Integer('Limit output data', required=False)

        with sdk2.parameters.Group('YT'):
            with sdk2.parameters.RadioGroup('YT Cluster') as yt_cluster:
                yt_cluster.values.hahn = yt_cluster.Value(value='hahn', default=True)
                yt_cluster.values.arnold = yt_cluster.Value(value='arnold')

            yt_path = sdk2.parameters.String('Target table', required=True)
            # noinspection SpellCheckingInspection
            yt_secret = sdk2.parameters.YavSecret('YAV secret YT token', default='sec-01d2ffwrdbwyj37zkj4r8zegsn')
            yt_secret_key = sdk2.parameters.String('Key to extract from YAV secret',
                                                   default_value='robot-quasar-yt-token')

        with sdk2.parameters.Group('YAV'):
            # noinspection SpellCheckingInspection
            yav_secret = sdk2.parameters.YavSecret('YAV secret with ssh key', default='sec-01dh4eer2d3pt81r5jfvezqefn')
            yav_secret_key = sdk2.parameters.String('Key to extract from YAV secret', default_value='yandex.key.txt')

    @property
    def binary_executor_query(self):
        return {"attrs": {'released': ctt.ReleaseStatus.STABLE, 'name': self.TASKS_RESOURCE_NAME}}

    @staticmethod
    def patch_fields(events):
        from dateutil.parser import isoparse
        from dateutil.tz import UTC

        def timestamp(utc_dt):
            return calendar.timegm(utc_dt.timetuple()) + float(utc_dt.microsecond) / 1000000

        for event in events:
            utc_dt = isoparse(event['timestamp']).astimezone(UTC).replace(tzinfo=None)
            event['timestamp'] = timestamp(utc_dt)

        return events

    @property
    def last_event_query(self):
        return 'SELECT timestamp FROM "{}" ORDER BY timestamp DESC LIMIT 1'.format(self.Parameters.yt_path)

    def get_last_event(self):
        import yt.clickhouse as chyt

        result = list(chyt.execute(self.last_event_query, alias='*ch_public'))
        return result[0] if result else None

    def upload_to_yt(self, data):
        import yt.wrapper as yt

        if yt.exists(self.Parameters.yt_path):
            table = yt.TablePath(self.Parameters.yt_path, append=True)
        else:
            table = yt.TablePath(self.Parameters.yt_path, append=False, attributes={'schema': EVENTS_SCHEMA})

        yt.write_table(table, data, format='json')

    @staticmethod
    def get_events_url(ts, limit=None):
        from furl import furl

        url = furl('http://localhost')
        url.port = 8080
        # noinspection PyPropertyAccess
        url.path = 'api/v1/device/events/'
        url.args['after_timestamp'] = '{:f}'.format(ts)
        # url.args['device_model'] = self.Parameters.device_model
        if limit is not None:
            url.args['limit'] = limit

        logging.info('requested url: %s', url.url)
        return url.url

    def calculate_ts_param(self):
        import yt.wrapper as yt

        ts = float(self.Parameters.after_timestamp)
        if ts > 0:
            return ts

        if yt.exists(self.Parameters.yt_path):
            event = self.get_last_event()
            if event:
                return event['timestamp']

        return ts

    def init_yt(self):
        import yt.wrapper as yt

        yt.config.set_proxy(self.Parameters.yt_cluster)
        yt.config['token'] = self.Parameters.yt_secret.data()[self.Parameters.yt_secret_key]

    @property
    def ssh_key(self):
        return self.Parameters.yav_secret.data()[self.Parameters.yav_secret_key]

    def write_key_to_file(self, filename):
        with open(filename, 'w') as f:
            f.write(self.ssh_key)
            f.flush()

    def on_execute(self):
        from sandbox.projects.quasar.jangles_lib import SshClient
        self.init_yt()

        # noinspection PyUnusedLocal
        events = None
        with tempfile.NamedTemporaryFile(delete=True) as ssh_key_file:
            self.write_key_to_file(ssh_key_file.name)
            ssh = SshClient(self.Parameters.host, self.Parameters.port, self.Parameters.user, ssh_key_file.name)
            ssh.forward_jangles_port_to(self.LOCAL_PORT)

            try:
                ts = self.calculate_ts_param()
                limit = self.Parameters.data_limit
                url = self.get_events_url(ts, limit)

                response = requests.get(url)
                logging.debug('jangles response code: %d', response.status_code)
                if response.status_code == 400:
                    logging.debug('client error, details: %s', response.json())

                response.raise_for_status()
                events = response.json()['events']
            except requests.exceptions.ConnectionError:
                # todo: use /ping to answer the question
                raise Exception('localhost connection error, jangles is down?')
            except requests.exceptions.HTTPError:
                raise Exception('invalid jangles response code')
            finally:
                ssh.terminate_connect()

        if events is None:
            logging.info('no new events')
            return

        yt_data = self.patch_fields(events)
        self.upload_to_yt(yt_data)

        logging.info('New %d events received: %s', len(events))
