import sys
import time
import json
import socket
import logging
import datetime
import posixpath

from random import shuffle

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.common.yasm import push_api
from sandbox.common.types.misc import DnsType


def report_name_comparator(first, second):
    """
    Compares filenames of two reports to sort them by loading priority, first to go first.
    Preliminary reports are sorted last, earliest reports sorted first.

    >>> sorted(['S.123.json', 'S.012.json'], cmp=report_name_comparator)
    ['S.012.json', 'S.123.json']
    >>> sorted(['S.123.preliminary.json', 'S.123.json'], cmp=report_name_comparator)
    ['S.123.json', 'S.123.preliminary.json']
    """

    def get_ts(filename):
        try:
            return int(filename.split('.')[1])
        except (ValueError, IndexError):
            return sys.maxint

    first_ts, second_ts = map(get_ts, [first, second])
    first_preliminary, second_preliminary = ['preliminary' in r for r in [first, second]]

    if (first_preliminary and second_preliminary) or (not first_preliminary and not second_preliminary):
        return first_ts - second_ts  # if same kind -- earliest timestamp goes first
    else:
        if first_preliminary:
            return 1  # if different kind -- non-preliminary go first
        else:
            return -1


class QuasarFetchChamberReports(sdk2.Task):
    """
    https://st.yandex-team.ru/QUASAR-1128

    Fetch reports from server @ 3nod and put em in YT
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment('yandex-yt', use_wheel=True),
            PipEnvironment('yandex-yt-yson-bindings-skynet', use_wheel=True),
        ]

        dns = DnsType.DNS64  # for external interactions
        cores = 1  # we are light task, only move some data over net

        class Caches(sdk2.Requirements.Caches):
            pass

    __DEFAULT_TTL = object()

    def _send_signal(self, name, value, ttl=__DEFAULT_TTL):
        """
        Sends a signal to YASM
        :param str name: as required by YASM API, see https://wiki.yandex-team.ru/golovan/push-api/#naming
        :param int ttl: signal ttl, see yasm doc. Defaults assume once-per-`run_time`-run
        :param value: a signle value point
        """

        if ttl is self.__DEFAULT_TTL:
            ttl = self.Parameters.run_time

        push_api.push_signal(name=name, value=value, prj='quasar-chamber', ttl=ttl)

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600  # should work at most 5 minutes, have some timeout here too

        run_time = sdk2.parameters.Integer('Run time, in seconds', default=300)

        with sdk2.parameters.Group('YT'):
            yt_proxy = sdk2.parameters.String('YT proxy', default='hahn')
            yt_base_path = sdk2.parameters.String('YT table base path', default='//home/quasar-dev/chamber/raw')

        with sdk2.parameters.Group('SFTP'):
            user = sdk2.parameters.String('SFTP user', default='yandex')
            host = sdk2.parameters.String('SFTP host', default='3nod-hkproxy.ext.eine.yandex.net')
            port = sdk2.parameters.Integer('SFTP server port', default=2222)
            archivate = sdk2.parameters.Bool('Move read reports to _old dir', default=True)
            reports_dir = sdk2.parameters.String('SFTP new reports dir', default='test_reports')
            reports_archive_dir = sdk2.parameters.String('SFTP archive reports dir', default='test_reports_old')

        with sdk2.parameters.Group('SSH'):
            ssh_port = sdk2.parameters.Integer('SSH server port', default=1222)
            ssh_user = sdk2.parameters.String('SSH user', default='yandex_telemetry')

        with sdk2.parameters.Group('Internal'):
            test = sdk2.parameters.Bool('Test some new and exciting feature', default=False)

    def _report_freespace(self):
        free_space = self.get_free_space()

        if free_space is not None:
            logging.info('Target has {%f} free space' % free_space)

            self._send_signal('server-free-space_nnnn', free_space)
        else:
            logging.warn('Failed to get free space')

    def _list_reports(self, sftp):
        """
        :returns: filenames of report files to be processed
        """
        logging.info('Listing...')

        all_reports = sftp.listdir(self.Parameters.reports_dir)

        logging.info('found %d reports', len(all_reports))

        self._send_signal('unprocessed-reports_xxxx', len(all_reports))

        return all_reports

    def _prepare_table(self):
        """
        :returns: pair (yt_client, yt_table_path) to write to
        """
        from yt.wrapper import YtClient, TablePath

        token = sdk2.Vault.data("robot-quasar-yt-token")
        client = YtClient("hahn", token)

        today = datetime.date.today().strftime('%Y%m%d')
        table = posixpath.join(self.Parameters.yt_base_path, today)

        # a schema for resulting table, matches fields in OpenHTF reports JSON
        schema = [
            {'name': n, 'type': v}
            for (n, v) in dict(
                dut_id='string',
                code_info='any',
                end_time_millis='int64',
                log_record='any',
                metadata='any',
                outcome='string',
                outcome_details='any',
                phases='any',
                start_time_millis='int64',
                station_id='string',
                log_records='any',
            ).items()
        ]

        table_path = TablePath(table, append=True)

        logging.info('Uploading to %s', table)

        if not client.exists(table_path):
            logging.info('It does not exist, creating..')
            client.create_table(table_path, attributes={'schema': schema})
        else:
            logging.info('It exists')

        return (client, table_path)

    def on_execute(self):
        self._report_freespace()

        sftp = self.connect_sftp()

        all_reports = self._list_reports(sftp)

        reports = sorted(all_reports, cmp=report_name_comparator)

        head = reports[:100]

        # shuffle head reports so bad reports wont be reloaded over-and-over again
        shuffle(head)

        reports[:100] = head

        if reports:
            end_time = time.time() + self.Parameters.run_time

            client, table_path = self._prepare_table()

            final_reports = filter(lambda r: '.preliminary.json' not in r, reports)
            processed = self._upload_reports(final_reports, reports, client, table_path, sftp, end_time, 'final')

            other_reports = filter(lambda r: r not in processed, reports)
            self._upload_reports(other_reports, reports, client, table_path, sftp, end_time, 'preliminary')

    def _upload_reports(self, reports, all_reports, yt_client, yt_table_path, sftp_client, end_time, prefix):
        """
        The 'do the work' function to process reports

        :param List[str] reports: of filenames to load
        :param List[str] all_reports: all reports @ server -- to check preliminaries
        :param YTClient yt_client: to upload via
        :param TablePath yt_table_path: to upload to
        :param SFTPClient sftp_client: to get from
        :param int end_time: max time to run, to prevent stucks
        :param str prefix: logging prefix

        :returns: list of reports processed (all what was moved to _old)
        """
        from paramiko import SSHException

        def archivate(a_report):
            original_path = posixpath.join(self.Parameters.reports_dir, a_report)
            archive_path = posixpath.join(self.Parameters.reports_archive_dir, a_report)
            logging.info('archiving report to %s', archive_path)
            sftp_client.rename(original_path, archive_path)

            # send a signal 'archived a report'
            self._send_signal('archived-reports_mmmm', value=1, ttl=None)

        processed = []

        for n, report in enumerate(reports, start=1):
            if time.time() > end_time:
                logging.info('End time of %d reached, exiting' % end_time)

                break
            try:
                path = posixpath.join(self.Parameters.reports_dir, report)
                size = sftp_client.stat(path).st_size

                logging.info('[%d of %d %s] processing report %s of %d bytes', n, len(reports), prefix, path, size)

                f = sftp_client.open(path)
                f.prefetch(size)  # prefetch asynchronously as we are reading it all!
                data = json.load(f)
                logging.info('Report file loaded')

                yt_client.write_table(yt_table_path, [data], format='json', table_writer={"max_row_weight": 100000000})

                # send a signal 'good report happened'
                self._send_signal('good-reports_mmmm', value=1, ttl=None)
                processed.append(report)

                if self.Parameters.archivate:
                    archivate(report)

                    # archivate preliminary too
                    preliminary_report = report.replace('.json', '.preliminary.json')

                    if preliminary_report in all_reports:
                        logging.info('Found preliminary report <%s>, archiving it without processing' % preliminary_report)

                        # send a signal 'good report happened' -- we treat preliminary as auto-success-loaded
                        self._send_signal('good-reports_mmmm', value=1, ttl=None)
                        processed.append(preliminary_report)

                        archivate(preliminary_report)

            except (SSHException, socket.error):
                logging.exception('Communication problem, aborting')
                break
            except Exception:
                logging.exception('Failed to load report %s', report)

                # send a signal 'bad report happened'
                self._send_signal('bad-reports_mmmm', value=1, ttl=None)

        return processed

    def _get_pkey(self, vault_entry='robot-quasar-3nod-sftp-key'):
        """
        :return: `paramiko.RSAKey` to be used for connections
        """
        import paramiko as p
        from StringIO import StringIO

        private_key = sdk2.Vault.data(vault_entry)
        key = p.RSAKey.from_private_key(StringIO(private_key))

        return key

    def connect_ssh(self):
        logging.info('connecting to SSH..')

        import paramiko as p

        ssh = p.SSHClient()
        ssh.set_missing_host_key_policy(p.client.WarningPolicy())  # FIXME: maybe not very good?
        ssh.connect(
            hostname=self.Parameters.host,
            port=self.Parameters.ssh_port,
            username=self.Parameters.ssh_user,
            pkey=self._get_pkey(vault_entry='robot-quasar-3nod-ssh-key'),
            compress=True
        )

        return ssh

    def connect_sftp(self):
        logging.info('connecting to SFTP..')

        import paramiko as p

        transport = p.Transport((self.Parameters.host, self.Parameters.port))

        transport.use_compression(True)
        transport.connect(username=self.Parameters.user, pkey=self._get_pkey())
        sftp = p.SFTPClient.from_transport(transport)
        sftp.get_channel().settimeout(60.0)  # 1 minute timeout

        logging.info('connected!')

        return sftp

    def get_free_space(self):
        """
        :return: free space on / in shares of 1 (e.g. 0 for no free space, 0.5 for half space used)
        """

        ssh = None

        try:
            ssh = self.connect_ssh()
            _, out, _ = ssh.exec_command('df /')

            # out looks like:
            #   Filesystem      Size  Used Avail Use% Mounted on
            #   /dev/md0        1.8T  1.3T  465G  74% /
            usage_percent = int(out.read().split('\n')[1].split()[4].replace('%', ''))
            return 1.0 - (usage_percent / 100.0)
        except Exception:
            logging.warn('Failed to get disk space: ', exc_info=True)
        finally:
            if ssh is not None:
                try:
                    ssh.close()
                except Exception:
                    logging.warn('Failed to close ssh connection: ', exc_info=True)
